/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.eureka;

import com.netflix.discovery.util.RateLimiter;
import com.netflix.eureka.EurekaServerConfig;
import com.netflix.eureka.EurekaServerContext;
import com.netflix.eureka.util.EurekaMonitors;
import jakarta.inject.Inject;
import jakarta.inject.Singleton;
import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class RateLimitingFilter
implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(RateLimitingFilter.class);
    private static final Set<String> DEFAULT_PRIVILEGED_CLIENTS = new HashSet<String>(Arrays.asList("DefaultClient", "DefaultServer"));
    private static final Pattern TARGET_RE = Pattern.compile("^.*/apps(/[^/]*)?$");
    private static final RateLimiter registryFetchRateLimiter = new RateLimiter(TimeUnit.SECONDS);
    private static final RateLimiter registryFullFetchRateLimiter = new RateLimiter(TimeUnit.SECONDS);
    private EurekaServerConfig serverConfig;

    @Inject
    public RateLimitingFilter(EurekaServerContext server) {
        this.serverConfig = server.getServerConfig();
    }

    public RateLimitingFilter() {
    }

    public void init(FilterConfig filterConfig) throws ServletException {
        if (this.serverConfig == null) {
            EurekaServerContext serverContext = (EurekaServerContext)filterConfig.getServletContext().getAttribute(EurekaServerContext.class.getName());
            this.serverConfig = serverContext.getServerConfig();
        }
    }

    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        Target target = RateLimitingFilter.getTarget(request);
        if (target == Target.Other) {
            chain.doFilter(request, response);
            return;
        }
        HttpServletRequest httpRequest = (HttpServletRequest)request;
        if (this.isRateLimited(httpRequest, target)) {
            this.incrementStats(target);
            if (this.serverConfig.isRateLimiterEnabled()) {
                ((HttpServletResponse)response).setStatus(503);
                return;
            }
        }
        chain.doFilter(request, response);
    }

    private static Target getTarget(ServletRequest request) {
        Target target = Target.Other;
        if (request instanceof HttpServletRequest) {
            Matcher matcher;
            HttpServletRequest httpRequest = (HttpServletRequest)request;
            String pathInfo = httpRequest.getRequestURI();
            if ("GET".equals(httpRequest.getMethod()) && pathInfo != null && (matcher = TARGET_RE.matcher(pathInfo)).matches()) {
                target = matcher.groupCount() == 0 || matcher.group(1) == null || "/".equals(matcher.group(1)) ? Target.FullFetch : ("/delta".equals(matcher.group(1)) ? Target.DeltaFetch : Target.Application);
            }
            if (target == Target.Other) {
                logger.debug("URL path {} not matched by rate limiting filter", (Object)pathInfo);
            }
        }
        return target;
    }

    private boolean isRateLimited(HttpServletRequest request, Target target) {
        if (this.isPrivileged(request)) {
            logger.debug("Privileged {} request", (Object)target);
            return false;
        }
        if (this.isOverloaded(target)) {
            logger.debug("Overloaded {} request; discarding it", (Object)target);
            return true;
        }
        logger.debug("{} request admitted", (Object)target);
        return false;
    }

    private boolean isPrivileged(HttpServletRequest request) {
        String clientName;
        if (this.serverConfig.isRateLimiterThrottleStandardClients()) {
            return false;
        }
        Set<String> privilegedClients = this.serverConfig.getRateLimiterPrivilegedClients();
        return privilegedClients.contains(clientName = request.getHeader("DiscoveryIdentity-Name")) || DEFAULT_PRIVILEGED_CLIENTS.contains(clientName);
    }

    private boolean isOverloaded(Target target) {
        int fetchWindowSize;
        boolean overloaded;
        int maxInWindow = this.serverConfig.getRateLimiterBurstSize();
        boolean bl = overloaded = !registryFetchRateLimiter.acquire(maxInWindow, (long)(fetchWindowSize = this.serverConfig.getRateLimiterRegistryFetchAverageRate()));
        if (target == Target.FullFetch) {
            int fullFetchWindowSize = this.serverConfig.getRateLimiterFullFetchAverageRate();
            overloaded |= !registryFullFetchRateLimiter.acquire(maxInWindow, (long)fullFetchWindowSize);
        }
        return overloaded;
    }

    private void incrementStats(Target target) {
        if (this.serverConfig.isRateLimiterEnabled()) {
            EurekaMonitors.RATE_LIMITED.increment();
            if (target == Target.FullFetch) {
                EurekaMonitors.RATE_LIMITED_FULL_FETCH.increment();
            }
        } else {
            EurekaMonitors.RATE_LIMITED_CANDIDATES.increment();
            if (target == Target.FullFetch) {
                EurekaMonitors.RATE_LIMITED_FULL_FETCH_CANDIDATES.increment();
            }
        }
    }

    public void destroy() {
    }

    static void reset() {
        registryFetchRateLimiter.reset();
        registryFullFetchRateLimiter.reset();
    }

    static enum Target {
        FullFetch,
        DeltaFetch,
        Application,
        Other;

    }
}

