/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.loadbalancer;

import com.google.common.base.Strings;
import com.netflix.client.ClientException;
import com.netflix.client.ClientRequest;
import com.netflix.client.DefaultLoadBalancerRetryHandler;
import com.netflix.client.IClientConfigAware;
import com.netflix.client.RetryHandler;
import com.netflix.client.config.CommonClientConfigKey;
import com.netflix.client.config.DefaultClientConfigImpl;
import com.netflix.client.config.IClientConfig;
import com.netflix.loadbalancer.AbstractLoadBalancer;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.LoadBalancerStats;
import com.netflix.loadbalancer.Server;
import com.netflix.loadbalancer.ServerStats;
import com.netflix.servo.monitor.Monitors;
import com.netflix.servo.monitor.Timer;
import com.netflix.util.Pair;
import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LoadBalancerContext
implements IClientConfigAware {
    private static final Logger logger = LoggerFactory.getLogger(LoadBalancerContext.class);
    protected String clientName = "default";
    protected String vipAddresses;
    protected int maxAutoRetriesNextServer = 1;
    protected int maxAutoRetries = 0;
    protected RetryHandler defaultRetryHandler = new DefaultLoadBalancerRetryHandler();
    protected boolean okToRetryOnAllOperations = DefaultClientConfigImpl.DEFAULT_OK_TO_RETRY_ON_ALL_OPERATIONS;
    private ILoadBalancer lb;
    private volatile Timer tracer;

    public LoadBalancerContext(ILoadBalancer lb) {
        this.lb = lb;
    }

    public LoadBalancerContext(ILoadBalancer lb, IClientConfig clientConfig) {
        this.lb = lb;
        this.initWithNiwsConfig(clientConfig);
    }

    public LoadBalancerContext(ILoadBalancer lb, IClientConfig clientConfig, RetryHandler handler) {
        this(lb, clientConfig);
        this.defaultRetryHandler = handler;
    }

    public void initWithNiwsConfig(IClientConfig clientConfig) {
        if (clientConfig == null) {
            return;
        }
        this.clientName = clientConfig.getClientName();
        if (this.clientName == null) {
            this.clientName = "default";
        }
        this.vipAddresses = clientConfig.resolveDeploymentContextbasedVipAddresses();
        this.maxAutoRetries = clientConfig.getPropertyAsInteger(CommonClientConfigKey.MaxAutoRetries, 0);
        this.maxAutoRetriesNextServer = clientConfig.getPropertyAsInteger(CommonClientConfigKey.MaxAutoRetriesNextServer, this.maxAutoRetriesNextServer);
        this.okToRetryOnAllOperations = clientConfig.getPropertyAsBoolean(CommonClientConfigKey.OkToRetryOnAllOperations, this.okToRetryOnAllOperations);
        this.defaultRetryHandler = new DefaultLoadBalancerRetryHandler(clientConfig);
        this.tracer = this.getExecuteTracer();
        Monitors.registerObject((String)("Client_" + this.clientName), (Object)this);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Timer getExecuteTracer() {
        if (this.tracer == null) {
            LoadBalancerContext loadBalancerContext = this;
            synchronized (loadBalancerContext) {
                if (this.tracer == null) {
                    this.tracer = Monitors.newTimer((String)(this.clientName + "_LoadBalancerExecutionTimer"), (TimeUnit)TimeUnit.MILLISECONDS);
                }
            }
        }
        return this.tracer;
    }

    public String getClientName() {
        return this.clientName;
    }

    public ILoadBalancer getLoadBalancer() {
        return this.lb;
    }

    public void setLoadBalancer(ILoadBalancer lb) {
        this.lb = lb;
    }

    @Deprecated
    public int getMaxAutoRetriesNextServer() {
        return this.maxAutoRetriesNextServer;
    }

    @Deprecated
    public void setMaxAutoRetriesNextServer(int maxAutoRetriesNextServer) {
        this.maxAutoRetriesNextServer = maxAutoRetriesNextServer;
    }

    @Deprecated
    public int getMaxAutoRetries() {
        return this.maxAutoRetries;
    }

    @Deprecated
    public void setMaxAutoRetries(int maxAutoRetries) {
        this.maxAutoRetries = maxAutoRetries;
    }

    protected Throwable getDeepestCause(Throwable e) {
        if (e != null) {
            int infiniteLoopPreventionCounter = 10;
            while (e.getCause() != null && infiniteLoopPreventionCounter > 0) {
                --infiniteLoopPreventionCounter;
                e = e.getCause();
            }
        }
        return e;
    }

    private boolean isPresentAsCause(Throwable throwableToSearchIn, Class<? extends Throwable> throwableToSearchFor) {
        return LoadBalancerContext.isPresentAsCauseHelper(throwableToSearchIn, throwableToSearchFor) != null;
    }

    static Throwable isPresentAsCauseHelper(Throwable throwableToSearchIn, Class<? extends Throwable> throwableToSearchFor) {
        for (int infiniteLoopPreventionCounter = 10; throwableToSearchIn != null && infiniteLoopPreventionCounter > 0; --infiniteLoopPreventionCounter, throwableToSearchIn = throwableToSearchIn.getCause()) {
            if (!throwableToSearchIn.getClass().isAssignableFrom(throwableToSearchFor)) continue;
            return throwableToSearchIn;
        }
        return null;
    }

    protected ClientException generateNIWSException(String uri, Throwable e) {
        ClientException niwsClientException = this.isPresentAsCause(e, SocketTimeoutException.class) ? this.generateTimeoutNIWSException(uri, e) : (e.getCause() instanceof UnknownHostException ? new ClientException(ClientException.ErrorType.UNKNOWN_HOST_EXCEPTION, "Unable to execute RestClient request for URI:" + uri, e) : (e.getCause() instanceof ConnectException ? new ClientException(ClientException.ErrorType.CONNECT_EXCEPTION, "Unable to execute RestClient request for URI:" + uri, e) : (e.getCause() instanceof NoRouteToHostException ? new ClientException(ClientException.ErrorType.NO_ROUTE_TO_HOST_EXCEPTION, "Unable to execute RestClient request for URI:" + uri, e) : (e instanceof ClientException ? (ClientException)e : new ClientException(ClientException.ErrorType.GENERAL, "Unable to execute RestClient request for URI:" + uri, e)))));
        return niwsClientException;
    }

    private boolean isPresentAsCause(Throwable throwableToSearchIn, Class<? extends Throwable> throwableToSearchFor, String messageSubStringToSearchFor) {
        Throwable throwableFound = LoadBalancerContext.isPresentAsCauseHelper(throwableToSearchIn, throwableToSearchFor);
        if (throwableFound != null) {
            return throwableFound.getMessage().contains(messageSubStringToSearchFor);
        }
        return false;
    }

    private ClientException generateTimeoutNIWSException(String uri, Throwable e) {
        ClientException niwsClientException = this.isPresentAsCause(e, SocketTimeoutException.class, "Read timed out") ? new ClientException(ClientException.ErrorType.READ_TIMEOUT_EXCEPTION, "Unable to execute RestClient request for URI:" + uri + ":" + this.getDeepestCause(e).getMessage(), e) : new ClientException(ClientException.ErrorType.SOCKET_TIMEOUT_EXCEPTION, "Unable to execute RestClient request for URI:" + uri + ":" + this.getDeepestCause(e).getMessage(), e);
        return niwsClientException;
    }

    private void recordStats(ServerStats stats, long responseTime) {
        stats.decrementActiveRequestsCount();
        stats.incrementNumRequests();
        stats.noteResponseTime(responseTime);
    }

    protected void noteRequestCompletion(ServerStats stats, Object response, Throwable e, long responseTime) {
        this.noteRequestCompletion(stats, response, e, responseTime, null);
    }

    public void noteRequestCompletion(ServerStats stats, Object response, Throwable e, long responseTime, RetryHandler errorHandler) {
        try {
            RetryHandler callErrorHandler;
            this.recordStats(stats, responseTime);
            RetryHandler retryHandler = callErrorHandler = errorHandler == null ? this.getRetryHandler() : errorHandler;
            if (callErrorHandler != null && response != null) {
                stats.clearSuccessiveConnectionFailureCount();
            } else if (callErrorHandler != null && e != null) {
                if (callErrorHandler.isCircuitTrippingException(e)) {
                    stats.incrementSuccessiveConnectionFailureCount();
                    stats.addToFailureCount();
                } else {
                    stats.clearSuccessiveConnectionFailureCount();
                }
            }
        }
        catch (Throwable ex) {
            logger.error("Unexpected exception", ex);
        }
    }

    protected void noteError(ServerStats stats, ClientRequest request, Throwable e, long responseTime) {
        try {
            this.recordStats(stats, responseTime);
            RetryHandler errorHandler = this.getRetryHandler();
            if (errorHandler != null && e != null) {
                if (errorHandler.isCircuitTrippingException(e)) {
                    stats.incrementSuccessiveConnectionFailureCount();
                    stats.addToFailureCount();
                } else {
                    stats.clearSuccessiveConnectionFailureCount();
                }
            }
        }
        catch (Throwable ex) {
            logger.error("Unexpected exception", ex);
        }
    }

    protected void noteResponse(ServerStats stats, ClientRequest request, Object response, long responseTime) {
        try {
            this.recordStats(stats, responseTime);
            RetryHandler errorHandler = this.getRetryHandler();
            if (errorHandler != null && response != null) {
                stats.clearSuccessiveConnectionFailureCount();
            }
        }
        catch (Throwable ex) {
            logger.error("Unexpected exception", ex);
        }
    }

    public void noteOpenConnection(ServerStats serverStats) {
        if (serverStats == null) {
            return;
        }
        try {
            serverStats.incrementActiveRequestsCount();
        }
        catch (Throwable e) {
            logger.info("Unable to note Server Stats:", e);
        }
    }

    protected Pair<String, Integer> deriveSchemeAndPortFromPartialUri(URI uri) {
        int port;
        boolean isSecure = false;
        String scheme = uri.getScheme();
        if (scheme != null) {
            isSecure = scheme.equalsIgnoreCase("https");
        }
        if ((port = uri.getPort()) < 0 && !isSecure) {
            port = 80;
        } else if (port < 0 && isSecure) {
            port = 443;
        }
        if (scheme == null) {
            scheme = isSecure ? "https" : "http";
        }
        return new Pair((Object)scheme, (Object)port);
    }

    protected int getDefaultPortFromScheme(String scheme) {
        if (scheme == null) {
            return -1;
        }
        if (scheme.equals("http")) {
            return 80;
        }
        if (scheme.equals("https")) {
            return 443;
        }
        return -1;
    }

    protected Pair<String, Integer> deriveHostAndPortFromVipAddress(String vipAddress) throws URISyntaxException, ClientException {
        String host;
        Pair hostAndPort = new Pair(null, (Object)-1);
        URI uri = new URI(vipAddress);
        String scheme = uri.getScheme();
        if (scheme == null) {
            uri = new URI("http://" + vipAddress);
        }
        if ((host = uri.getHost()) == null) {
            throw new ClientException("Unable to derive host/port from vip address " + vipAddress);
        }
        int port = uri.getPort();
        if (port < 0) {
            port = this.getDefaultPortFromScheme(scheme);
        }
        if (port < 0) {
            throw new ClientException("Unable to derive host/port from vip address " + vipAddress);
        }
        hostAndPort.setFirst((Object)host);
        hostAndPort.setSecond((Object)port);
        return hostAndPort;
    }

    private boolean isVipRecognized(String vipEmbeddedInUri) {
        String[] addresses;
        if (vipEmbeddedInUri == null) {
            return false;
        }
        if (this.vipAddresses == null) {
            return false;
        }
        for (String address : addresses = this.vipAddresses.split(",")) {
            if (!vipEmbeddedInUri.equalsIgnoreCase(address.trim())) continue;
            return true;
        }
        return false;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public Server getServerFromLoadBalancer(@Nullable URI original, @Nullable Object loadBalancerKey) throws ClientException {
        String host = null;
        int port = -1;
        if (original != null) {
            host = original.getHost();
        }
        if (original != null) {
            Pair<String, Integer> schemeAndPort = this.deriveSchemeAndPortFromPartialUri(original);
            port = (Integer)schemeAndPort.second();
        }
        ILoadBalancer lb = this.getLoadBalancer();
        if (host == null) {
            if (lb != null) {
                Server svc = lb.chooseServer(loadBalancerKey);
                if (svc == null) {
                    throw new ClientException(ClientException.ErrorType.GENERAL, "Load balancer does not have available server for client: " + this.clientName);
                }
                host = svc.getHost();
                if (host == null) {
                    throw new ClientException(ClientException.ErrorType.GENERAL, "Invalid Server for :" + svc);
                }
                logger.debug("{} using LB returned Server: {} for request {}", new Object[]{this.clientName, svc, original});
                return svc;
            }
            if (this.vipAddresses != null && this.vipAddresses.contains(",")) {
                throw new ClientException(ClientException.ErrorType.GENERAL, "Method is invoked for client " + this.clientName + " with partial URI of (" + original + ") with no load balancer configured." + " Also, there are multiple vipAddresses and hence no vip address can be chosen" + " to complete this partial uri");
            }
            if (this.vipAddresses == null) throw new ClientException(ClientException.ErrorType.GENERAL, this.clientName + " has no LoadBalancer registered and passed in a partial URL request (with no host:port)." + " Also has no vipAddress registered");
            try {
                Pair<String, Integer> hostAndPort = this.deriveHostAndPortFromVipAddress(this.vipAddresses);
                host = (String)hostAndPort.first();
                port = (Integer)hostAndPort.second();
            }
            catch (URISyntaxException e) {
                throw new ClientException(ClientException.ErrorType.GENERAL, "Method is invoked for client " + this.clientName + " with partial URI of (" + original + ") with no load balancer configured. " + " Also, the configured/registered vipAddress is unparseable (to determine host and port)");
            }
        } else {
            boolean shouldInterpretAsVip = false;
            if (lb != null) {
                shouldInterpretAsVip = this.isVipRecognized(original.getAuthority());
            }
            if (shouldInterpretAsVip) {
                Server svc = lb.chooseServer(loadBalancerKey);
                if (svc != null) {
                    host = svc.getHost();
                    if (host == null) {
                        throw new ClientException(ClientException.ErrorType.GENERAL, "Invalid Server for :" + svc);
                    }
                    logger.debug("using LB returned Server: {} for request: {}", (Object)svc, (Object)original);
                    return svc;
                }
                logger.debug("{}:{} assumed to be a valid VIP address or exists in the DNS", (Object)host, (Object)port);
            } else {
                logger.debug("Using full URL passed in by caller (not using load balancer): {}", (Object)original);
            }
        }
        if (host != null) return new Server(host, port);
        throw new ClientException(ClientException.ErrorType.GENERAL, "Request contains no HOST to talk to");
    }

    public URI reconstructURIWithServer(Server server, URI original) {
        String host = server.getHost();
        int port = server.getPort();
        if (host.equals(original.getHost()) && port == original.getPort()) {
            return original;
        }
        String scheme = original.getScheme();
        if (scheme == null) {
            scheme = (String)this.deriveSchemeAndPortFromPartialUri(original).first();
        }
        try {
            StringBuilder sb = new StringBuilder();
            sb.append(scheme).append("://");
            if (!Strings.isNullOrEmpty((String)original.getRawUserInfo())) {
                sb.append(original.getRawUserInfo()).append("@");
            }
            sb.append(host);
            if (port >= 0) {
                sb.append(":").append(port);
            }
            sb.append(original.getRawPath());
            if (!Strings.isNullOrEmpty((String)original.getRawQuery())) {
                sb.append("?").append(original.getRawQuery());
            }
            if (!Strings.isNullOrEmpty((String)original.getRawFragment())) {
                sb.append("#").append(original.getRawFragment());
            }
            URI newURI = new URI(sb.toString());
            return newURI;
        }
        catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    protected int getRetriesNextServer(IClientConfig overriddenClientConfig) {
        int numRetries = this.maxAutoRetriesNextServer;
        if (overriddenClientConfig != null) {
            numRetries = overriddenClientConfig.getPropertyAsInteger(CommonClientConfigKey.MaxAutoRetriesNextServer, this.maxAutoRetriesNextServer);
        }
        return numRetries;
    }

    public final ServerStats getServerStats(Server server) {
        ServerStats serverStats = null;
        ILoadBalancer lb = this.getLoadBalancer();
        if (lb instanceof AbstractLoadBalancer) {
            LoadBalancerStats lbStats = ((AbstractLoadBalancer)lb).getLoadBalancerStats();
            serverStats = lbStats.getSingleServerStat(server);
        }
        return serverStats;
    }

    protected int getNumberRetriesOnSameServer(IClientConfig overriddenClientConfig) {
        int numRetries = this.maxAutoRetries;
        if (overriddenClientConfig != null) {
            try {
                numRetries = overriddenClientConfig.getPropertyAsInteger(CommonClientConfigKey.MaxAutoRetries, this.maxAutoRetries);
            }
            catch (Exception e) {
                logger.warn("Invalid maxRetries requested for RestClient:" + this.clientName);
            }
        }
        return numRetries;
    }

    public boolean handleSameServerRetry(Server server, int currentRetryCount, int maxRetries, Throwable e) {
        if (currentRetryCount > maxRetries) {
            return false;
        }
        logger.debug("Exception while executing request which is deemed retry-able, retrying ..., SAME Server Retry Attempt#: {}", (Object)currentRetryCount, (Object)server);
        return true;
    }

    public final RetryHandler getRetryHandler() {
        return this.defaultRetryHandler;
    }

    public final void setRetryHandler(RetryHandler retryHandler) {
        this.defaultRetryHandler = retryHandler;
    }

    public final boolean isOkToRetryOnAllOperations() {
        return this.okToRetryOnAllOperations;
    }

    public final void setOkToRetryOnAllOperations(boolean okToRetryOnAllOperations) {
        this.okToRetryOnAllOperations = okToRetryOnAllOperations;
    }
}

