package com.centit.framework.hibernate.common;


import org.apache.commons.codec.binary.Hex;
import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SharedSessionContractImplementor;
import org.hibernate.usertype.UserType;

import java.io.*;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

/**
 * <p>
 *  自定义hibernate数据类型
 *  在Hibernate in Action一书中讲自定义hibernate数据类型因成为开发者的基础技能。
 *  个人感觉在hibernate中有多少种数据类型还没搞清楚了，还去自定义数据类型。。。。
 *
 *  Object序列化/反序列化
 *  数据库中以hex字符串存储
 *
 *  使用示例
 *  将@Type(type="com.centit.oa.common.hibernate.type.ObjectSerializeUserType")注解到bean的属性或get上去
 *
 * </p>
 *
 * @author AllanLoo
 * @version 1.0 2016-04-14
 */
public class ObjectSerializeUserType implements UserType, Serializable {
    private static final long serialVersionUID = -6712023745050775585L;

    @Override
    public int[] sqlTypes() {
        return new int[]{Types.VARCHAR};
    }

    @Override
    public Class<?> returnedClass() {
        return Object.class;
    }

    @Override
    public boolean equals(Object o, Object o1) throws HibernateException {
        if (o == o1) {
            return true;
        }
        if (o == null || o1 == null) {
            return false;
        }

        return o.equals(o1);
    }

    @Override
    public int hashCode(Object o) throws HibernateException {
        return o.hashCode();
    }

    /**
     *  从JDBC ResultSet读取数据,将其转换为自定义类型后返回
     * (此方法要求对克能出现null值进行处理)
     * names中包含了当前自定义类型的映射字段名称
     *
     * @param rs ResultSet
     * @param names String[]
     * @param session SharedSessionContractImplementor
     * @param owner Object
     * @return nullSafe
     * @throws HibernateException HibernateException
     * @throws SQLException SQLException
     */
    @Override
    public Object nullSafeGet(ResultSet rs, String[] names,
                              SharedSessionContractImplementor session, Object owner)
            throws HibernateException, SQLException{
        String hexStr = rs.getString(names[0]);
        try (ByteArrayInputStream bis = new ByteArrayInputStream(Hex.decodeHex(hexStr.toCharArray()));
             ObjectInputStream ois = new ObjectInputStream(bis)) {
             return ois.readObject();
        } catch (Exception e) {
            throw new HibernateException(e);
        }
    }

    /**
     * 本方法将在Hibernate进行数据保存时被调用
     * 我们可以通过PreparedStateme将自定义数据写入到对应的数据库表字段
     * @param st PreparedStatement
     * @param value Object
     * @param index int
     * @param session SharedSessionContractImplementor
     * @throws HibernateException HibernateException
     * @throws SQLException SQLException
     */
    @Override
    public void nullSafeSet(PreparedStatement st, Object value, int index,
                            SharedSessionContractImplementor session) throws HibernateException, SQLException{

        if (value == null) {
            st.setNull(index, Types.VARCHAR);
        } else {
            try(ByteArrayOutputStream bos = new ByteArrayOutputStream();
                ObjectOutputStream oos = new ObjectOutputStream(bos)) {
                oos.writeObject(value);
                byte[] objectBytes = bos.toByteArray();
                String hexStr = Hex.encodeHexString(objectBytes);
                st.setString(index, hexStr);
            } catch (Exception e) {
                throw new HibernateException(e);
            }
        }
    }

    /**
     * 提供自定义类型的完全复制方法
     * 本方法将用构造返回对象
     * 当nullSafeGet方法调用之后，我们获得了自定义数据对象，在向用户返回自定义数据之前，
     * deepCopy方法将被调用，它将根据自定义数据对象构造一个完全拷贝，并将此拷贝返回给用户
     * 此时我们就得到了自定义数据对象的两个版本，第一个是从数据库读出的原始版本，其二是我们通过
     * deepCopy方法构造的复制版本，原始的版本将有Hibernate维护，复制版由用户使用。原始版本用作
     * 稍后的脏数据检查依据；Hibernate将在脏数据检查过程中将两个版本的数据进行对比（通过调用
     * equals方法），如果数据发生了变化（equals方法返回false），则执行对应的持久化操作
     *
     * @param o Object
     * @return  deepCopy
     * @throws HibernateException HibernateException
     */
    @Override
    public Object deepCopy(Object o) throws HibernateException {
        if (o == null)
            return null;
        return o;
    }


    /**
     * 本类型实例是否可变
     *
     * @return 本类型实例是否可变
     */
    @Override
    public boolean isMutable() {
        return true;
    }

    /* 序列化 */
    @Override
    public Serializable disassemble(Object value) throws HibernateException {
        return ((Serializable) value);
    }

    /* 反序列化 */
    @Override
    public Object assemble(Serializable cached, Object o) throws HibernateException {
        return cached;
    }

    @Override
    public Object replace(Object original, Object target, Object owner) throws HibernateException {
        return original;
    }
}