package com.centit.framework.system.controller;

import com.alibaba.fastjson.JSONArray;
import com.centit.framework.common.JsonResultUtils;
import com.centit.framework.common.ResponseData;
import com.centit.framework.common.ResponseMapData;
import com.centit.framework.common.ViewDataTransform;
import com.centit.framework.core.controller.BaseController;
import com.centit.framework.model.adapter.PlatformEnvironment;
import com.centit.framework.operationlog.RecordOperationLog;
import com.centit.framework.system.po.OptInfo;
import com.centit.framework.system.po.OptMethod;
import com.centit.framework.system.service.OptInfoManager;
import com.centit.framework.system.service.OptMethodManager;
import com.centit.support.json.JsonPropertyUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.IOException;
import java.util.List;
import java.util.Map;

@Controller
@RequestMapping("/optinfo")
public class OptInfoController extends BaseController {
    @Resource
    private OptInfoManager optInfoManager;

    @Resource
    private OptMethodManager optMethodManager;

    @Resource
    private PlatformEnvironment platformEnvironment;

    /**
     * 系统日志中记录
     * @return 业务标识ID
     */
    //private String optId = "OPTINFO";//CodeRepositoryUtil.getCode("OPTID", "optInfo");
    public String getOptId() {
        return "OPTINFO";
    }

    /**
     * 查询所有系统业务
     *
     * @param id       父id
     * @param request  HttpServletRequest
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/sub", method = RequestMethod.GET)
    public void listFromParent(String id, HttpServletRequest request, HttpServletResponse response) {
        Map<String, Object> searchColumn = BaseController.convertSearchColumn(request);

        if (StringUtils.isNotBlank(id)) {
            searchColumn.put("preOptId", id);
        } else {
            searchColumn.put("NP_TOPOPT", "true");
        }

        List<OptInfo> listObjects = optInfoManager.listObjects(searchColumn);

        for (OptInfo opt : listObjects) {
            opt.setState(optInfoManager.hasChildren(opt.getOptId()) ? "closed" : "open");
        }
        JsonResultUtils.writeSingleDataJson(makeMenuFuncsJson(listObjects), response);
    }

    private JSONArray makeMenuFuncsJson(List<OptInfo> menuFunsByUser) {
        return ViewDataTransform.makeTreeViewJson(menuFunsByUser,
            ViewDataTransform.createStringHashMap("id", "optId",
                "optId", "optId",
                "optCode", "optId",
                "pid", "preOptId",
                "text", "optName",
                "url", "optRoute",
                "icon", "icon",
                "children", "children",
                "isInToolbar", "isInToolbar",
                "state", "state",
                "optMethods", "optMethods"
            ), (jsonObject, obj) -> jsonObject.put("external", !("D".equals(obj.getPageType()))));
    }

    /**
     * 查询所有需要通过权限管理的业务
     *
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/poweropts", method = RequestMethod.GET)
    public void listPowerOpts(HttpServletResponse response) {
        List<OptInfo> listObjects = optInfoManager.listSysAndOptPowerOpts();
        listObjects = optInfoManager.listObjectFormatTree(listObjects, true);
        JsonResultUtils.writeSingleDataJson(makeMenuFuncsJson(listObjects), response);
    }


    /**
     * 查询所有项目权限管理的业务
     *
     * @param field    需要显示的字段
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/itempoweropts", method = RequestMethod.GET)
    public void listItemPowerOpts(String[] field, HttpServletResponse response) {
        List<OptInfo> listObjects = optInfoManager.listItemPowerOpts();
        listObjects = optInfoManager.listObjectFormatTree(listObjects, true);

        if (ArrayUtils.isNotEmpty(field))
            JsonResultUtils.writeSingleDataJson(listObjects, response,
                JsonPropertyUtils.getIncludePropPreFilter(OptInfo.class, field));
        else
            JsonResultUtils.writeSingleDataJson(listObjects, response);
    }


    /**
     * 查询某个部门权限的业务
     *
     * @param field    需要显示的字段
     * @param unitCode unitCode
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/unitpoweropts/{unitCode}", method = RequestMethod.GET)
    public void listUnitPowerOpts(@PathVariable String unitCode, String[] field,
                                  HttpServletResponse response) {
        List<OptInfo> listObjects = optInfoManager.listOptWithPowerUnderUnit(unitCode);
        listObjects = optInfoManager.listObjectFormatTree(listObjects, false);

        JsonResultUtils.writeSingleDataJson(makeMenuFuncsJson(listObjects), response);
    }

    /**
     * 新增菜单
     *
     * @param optInfo  OptInfo
     * @param request  HttpServletRequest
     * @param response HttpServletResponse
     */
    @RequestMapping(method = {RequestMethod.POST})
    @RecordOperationLog(content = "操作IP地址:{userInfo.loginIp},用户{userInfo.userName}新增菜单")
    public void createOptInfo(@Valid OptInfo optInfo, HttpServletRequest request, HttpServletResponse response) {

        optInfoManager.saveNewOptInfo(optInfo);
        //刷新缓存
        JsonResultUtils.writeSingleDataJson(optInfo, response);
    }

    /**
     * optId是否已存在
     *
     * @param optId    optId
     * @param response HttpServletResponse
     * @throws IOException IOException
     */
    @RequestMapping(value = "/notexists/{optId}", method = {RequestMethod.GET})
    public void isNotExists(@PathVariable String optId, HttpServletResponse response) throws IOException {
        OptInfo optInfo = optInfoManager.getObjectById(optId);
        JsonResultUtils.writeOriginalObject(null == optInfo, response);
    }

    /**
     * 更新菜单
     *
     * @param optId    主键
     * @param optInfo  OptInfo
     * @param request  HttpServletRequest
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/{optId}", method = {RequestMethod.PUT})
    @RecordOperationLog(content = "操作IP地址:{userInfo.loginIp},用户{userInfo.userName}更新菜单")
    public void edit(@PathVariable String optId, @Valid OptInfo optInfo,
                     HttpServletRequest request, HttpServletResponse response) {

        OptInfo dbOptInfo = optInfoManager.getObjectById(optId);
        if (null == dbOptInfo) {
            JsonResultUtils.writeErrorMessageJson("当前对象不存在", response);
            return;
        }

        if (!StringUtils.equals(dbOptInfo.getPreOptId(), optInfo.getPreOptId())) {
            OptInfo parentOpt = optInfoManager.getOptInfoById(optInfo.getPreOptId());
            if (parentOpt == null) {
                optInfo.setPreOptId(dbOptInfo.getPreOptId());
            }
        }

        dbOptInfo.copyNotNullProperty(optInfo);
        //BeanUtils.copyProperties(optInfo, dbOptInfo, "optMethods", "dataScopes");
        optInfoManager.updateOptInfo(dbOptInfo);

        JsonResultUtils.writeSingleDataJson(dbOptInfo, response);
    }

    /**
     * 更新操作权限
     *
     * @param optId    主键
     * @param optInfo  OptInfo
     * @param request  HttpServletRequest
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/editpower{optId}", method = {RequestMethod.PUT})
    @RecordOperationLog(content = "操作IP地址:{userInfo.loginIp},用户{userInfo.userName}更新操作权限")
    public void editPower(@PathVariable String optId, @Valid OptInfo optInfo,
                          HttpServletRequest request, HttpServletResponse response) {

        OptInfo dbOptInfo = optInfoManager.getObjectById(optId);
        if (null == dbOptInfo) {
            JsonResultUtils.writeErrorMessageJson("当前对象不存在", response);
            return;
        }

        if (!StringUtils.equals(dbOptInfo.getPreOptId(), optInfo.getPreOptId())) {
            OptInfo parentOpt = optInfoManager.getOptInfoById(optInfo.getPreOptId());
            if (parentOpt == null) {
                optInfo.setPreOptId(dbOptInfo.getPreOptId());
            }
        }

        for (OptMethod optDef : optInfo.getOptMethods()) {
            if (StringUtils.isBlank(optDef.getOptCode())) {
                optDef.setOptCode(optMethodManager.getNextOptCode());
            }
        }

        dbOptInfo.copyNotNullProperty(optInfo);

        dbOptInfo.addAllOptMethods(optInfo.getOptMethods());
        dbOptInfo.addAllDataScopes(optInfo.getDataScopes());
        optInfoManager.updateOperationPower(dbOptInfo);
        JsonResultUtils.writeSuccessJson(response);
    }

    /**
     * 删除菜单
     *
     * @param optId    主键
     * @param request  HttpServletRequest
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/{optId}", method = {RequestMethod.DELETE})
    @RecordOperationLog(content = "操作IP地址:{userInfo.loginIp},用户{userInfo.userName}删除菜单")
    public void delete(@PathVariable String optId, HttpServletRequest request, HttpServletResponse response) {
        OptInfo dboptInfo = optInfoManager.getObjectById(optId);

        optInfoManager.deleteOptInfo(dboptInfo);

        JsonResultUtils.writeBlankJson(response);
    }

    /**
     * 查询单条数据
     *
     * @param optId    主键
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/{optId}", method = {RequestMethod.GET})
    public void getOptInfoById(@PathVariable String optId, HttpServletResponse response) {
        OptInfo dbOptInfo = optInfoManager.getOptInfoById(optId);

        JsonResultUtils.writeSingleDataJson(dbOptInfo, response);
    }

    /**
     * 新增页面时获取OptDef主键
     *
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/nextOptCode", method = RequestMethod.GET)
    public void getNextOptCode(HttpServletResponse response) {
        String optCode = optMethodManager.getNextOptCode();

        ResponseMapData responseData = new ResponseMapData();
        responseData.addResponseData("optCode", optCode);

        JsonResultUtils.writeResponseDataAsJson(responseData, response);
    }

    /**
     * 新建或更新业务操作
     *
     * @param optId    主键
     * @param optCode  optCode
     * @param optDef   OptMethod
     * @param response HttpServletResponse
     */
    @RequestMapping(value = "/{optId}/{optCode}", method = {RequestMethod.POST, RequestMethod.PUT})
    public void optDefEdit(@PathVariable String optId, @PathVariable String optCode, @Valid OptMethod optDef,
                           HttpServletResponse response) {
        OptInfo optInfo = optInfoManager.getObjectById(optId);
        if (null == optInfo) {
            JsonResultUtils.writeSingleErrorDataJson(
                ResponseData.ERROR_INTERNAL_SERVER_ERROR,
                "数据库不匹配", "数据库中不存在optId为" + optId + "的业务信息。", response);
            return;
        }

        OptMethod dbOptDef = optMethodManager.getObjectById(optCode);
        if (null == dbOptDef) {
            JsonResultUtils.writeSingleErrorDataJson(
                ResponseData.ERROR_INTERNAL_SERVER_ERROR,
                "数据库不匹配", "数据库中不存在optCode为" + optCode + "的操作信息。", response);
            return;
        } else {
            dbOptDef.copy(optDef);
            optMethodManager.updateOptMethod(dbOptDef);
        }

        JsonResultUtils.writeSuccessJson(response);
    }

    @RequestMapping(value = "/allOptInfo", method = RequestMethod.GET)
    public void loadAllOptInfo(HttpServletResponse response) {
        List<OptInfo> optInfos = optInfoManager.listObjects();
        JsonResultUtils.writeSingleDataJson(optInfos, response);
    }

    @RequestMapping(value = "/allOptMethod", method = RequestMethod.GET)
    public void loadAllOptMethod(HttpServletResponse response) {
        List<OptMethod> optDefs = optMethodManager.listObjects();
        JsonResultUtils.writeSingleDataJson(optDefs, response);
    }

    @RequestMapping(value = "/userpoweropts/{userCode}", method = RequestMethod.GET)
    public void listUserOpts(@PathVariable String userCode, HttpServletResponse response) {
//        List<OptInfo> optInfos = (List<OptInfo>) platformEnvironment.listUserMenuOptInfos(userCode, false);
        List<OptInfo> optInfos = optInfoManager.listUserAllPower(userCode, false);
        optInfos = optInfoManager.listObjectFormatTree(optInfos, true);
        JsonResultUtils.writeSingleDataJson(makeMenuFuncsJson(optInfos), response);
    }

}
