package com.contentsquare.android;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.Strings;

import java.util.Locale;

import hugo.weaving.DebugLog;

/**
 * This is the ContentSquare SDK main entry point.
 * please use this class to startSession your project for tracking.
 */
public class ContentSquare {
    private static final String TAG = "ContentSquare";
    private static final Logger LOG = new Logger(TAG);

    @VisibleForTesting
    ContentSquare() {
        //to assure that this class is not instantiated
        throw new UnsupportedOperationException("This class cannot be instantiated.");
    }

    /**
     * Starts the ContentSquare UX tracking SDK.
     *
     * @param context   - application context, the SDK will not start if a null context is
     *                  provided.
     * @param projectId - a non null String, representing the project ID registered with
     *                  Contentsquare. If you don't have one please contact your contact.
     */
    @DebugLog
    public static void startWithProjectId(@NonNull Context context, @NonNull String projectId) {
        if (context == null || Strings.isNullOrEmpty(projectId)) {
            //CHECKSTYLE:OFF
            android.util.Log.e(TAG,
                    String.format("Cannot start ContentSquare SDK, context = %s, projectId = %s",
                            context, projectId));
            //CHECKSTYLE:ON
        } else {
            LOG.d("Initializing singletons and starting run..");
            boolean initSuccessfullyRan = SingletonProvider.init(context, projectId);
            if (initSuccessfullyRan) {
                logInit(projectId);
                Session session = SingletonProvider.getSessionComponent().getSession();
                session.startSession();
            } else {
                //CHECKSTYLE:OFF
                android.util.Log.i(TAG, "SDK was pre-initialized, skipping call.");
                //CHECKSTYLE:ON
            }
        }
    }

    private static void logInit(@NonNull String projectId) {
        //CHECKSTYLE:OFF
        final long sdkTimestamp = BuildConfig.SDK_TIMESTAMP;
        final String format = "Starting ContentSquare SDK released with projectId = %s t: %d";
        final String initLog = String.format(Locale.ENGLISH,
                format,
                projectId,
                sdkTimestamp);
        android.util.Log.i(TAG, initLog);
        //CHECKSTYLE:ON
    }

    /**
     * Stops the ContentSquare UX tracking SDK.
     * Once this is called the SDK will completely stop.
     */
    public static void stopTracking() {
        //CHECKSTYLE:OFF
        android.util.Log.i(TAG, "Stopping ContentSquare Tracker.");
        //CHECKSTYLE:ON

        Session session = SingletonProvider.getSessionComponent().getSession();
        session.stopSession();
    }

    /**
     * Stops the ContentSquare UX tracking SDK.
     * Once this is called the SDK will completely stop.
     */
    public static void pauseTracking() {
        //CHECKSTYLE:OFF
        android.util.Log.i(TAG, "Pausing ContentSquare Tracker.");
        //CHECKSTYLE:ON
        Session session = SingletonProvider.getSessionComponent().getSession();
        session.pauseSession();
    }

    /**
     * Re-enables the ContentSquare UX tracking SDK.
     * This call will
     */
    public static void resumeTracking() {
        //CHECKSTYLE:OFF
        android.util.Log.i(TAG, "Un-pausing ContentSquare Tracker.");
        //CHECKSTYLE:ON
        Session session = SingletonProvider.getSessionComponent().getSession();
        session.resumeSession();
    }

}
