package com.contentsquare.android.internal;

import android.support.annotation.CheckResult;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.contentsquare.android.internal.config.ProjectConfiguration;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.Strings;
import com.contentsquare.android.internal.validator.InSampleIntervalValidator;

import org.json.JSONException;
import org.json.JSONObject;

import hugo.weaving.DebugLog;

/**
 * Compositor for runtime configurations.
 * This class will make a Composite {@link ProjectConfiguration}
 * using a composition technique to derive with a single configuration from:
 * <ul>
 * <li>Local default config</li>
 * <li>local cached base config</li>
 * <li>remote base config</li>
 * <li>local cached client config</li>
 * <li>remote client config</li>
 * </ul>
 * The composition will be of the kind "best effort" meaning, it will composite any config files
 * it's provided with and provide the most up to date config available under the runtime
 * environment.
 */
public class ConfigurationCompositor {
    //CHECKSTYLE:OFF
    // Project related labels
    public static final String CS_PROJECT_ID = "cs_project_id";
    public static final String CONFIG_VERSION = "version";

    // Tracker related labels
    public static final String TRACKER_ACTIVE = "enabled";
    public static final String ANONYMIZE = "anonymize";
    public static final String TRACKER_ENDPOINT_URL = "endpoint";
    public static final String SAMPLE_RATE = "sample";
    // Storage related labels
    public static final String STORAGE_MAX_ITEMS = "bucket";
    // Screen recording related labels
    public static final String SCREEN_RECORD_ACTIVE = "screenshots";
    // Debug related labels
    public static final String DEBUG_FLAG = "debug_flag";
    public static final String BLACKLISTED_VERSIONS = "blacklisted";
    private static final Logger LOGGER = new Logger("ConfigurationCompositor");
    //CHECKSTYLE:ON

    @NonNull
    private final InSampleIntervalValidator mInSampleValidator;

    /**
     * Creates a new instance of {@link ConfigurationCompositor}.
     *
     * @param sampleIntervalValidator as {@link InSampleIntervalValidator}
     */
    public ConfigurationCompositor(@NonNull InSampleIntervalValidator sampleIntervalValidator) {
        this.mInSampleValidator = sampleIntervalValidator;
    }

    /**
     * Creates a {@link ProjectConfiguration} from a list of configuration files (in Json format).
     * If the provided list is null, the response will be an empty (default) run-config.
     *
     * @param data a String which is representing a configuration.
     * @return a {@link ProjectConfiguration} which is a composition of all of the configs.
     */
    @NonNull
    public ProjectConfiguration getRunConfiguration(@Nullable String... data) {
        final ProjectConfiguration.Builder builder = ProjectConfiguration.builder();
        if (data != null) {

            for (String config : data) {
                updateBuilderBasedOnJsonConfig(config, builder);
            }
        } else {
            LOGGER.d("No data passed to config");
        }

        return builder.build();
    }

    /**
     * Updates a single configuration JSON to an existing {@link ProjectConfiguration}.
     * Provided both objects.
     *
     * @param config - an existing RunConfiguration
     * @param data   - the JSON configuration to update with
     * @return - the updated RunConfiguration.
     */
    @DebugLog
    @NonNull
    @CheckResult
    @SuppressWarnings("squid:S00112")
    public ProjectConfiguration updateRunConfigurationFromConfig(
            @NonNull ProjectConfiguration config,
            @Nullable String data) {

        final ProjectConfiguration.Builder builder = ProjectConfiguration.builder();
        builder.csProjectId(config.getCsProjectId())
                .configVersion(config.getConfigVersion())
                .trackingEnabled(config.isTrackingEnabled())
                .screenShotsEnabled(config.isScreenShotEnabled())
                .anonymize(config.isAnonymousEnabled())
                .collectorEndpoint(config.getCollectorEndpoint())
                .samplingRate(config.getTrackingSampleRate())
                .debugFlag(config.getGodModeKey())
                .maxBucketSize(config.getMaxBucketSize())
                .csProjectId(config.getCsProjectId())
                .blackListedVersionsList(config.getBlackListedVersionsList());
        updateBuilderBasedOnJsonConfig(data, builder);
        return builder.build();
    }

    /**
     * Validates a config JSON format whether is matching or not the new format.
     *
     * @param config as {@link JSONObject} to {@link String}
     * @return true if this config matches the new config format
     */
    public boolean validateConfig(@NonNull String config) {
        return config.contains(CS_PROJECT_ID);
    }

    private void updateBuilderBasedOnJsonConfig(
            @Nullable final String data, @NonNull final ProjectConfiguration.Builder builder) {
        LOGGER.d("updating config with : %s", Strings.nullToEmpty(data));
        JSONObject object;
        if (Strings.isNullOrEmpty(data)) {
            return;
        }

        try {
            object = new JSONObject(data);

            // project id
            builder.csProjectId(opt(object, CS_PROJECT_ID, builder.getCsProjectId()));

            // version
            builder.configVersion(opt(object, CONFIG_VERSION, builder.getVersion()));


            // tracker on?
            builder.trackingEnabled(object.optBoolean(TRACKER_ACTIVE, builder.isTrackerEnabled()));


            // should anonymize
            builder.anonymize(object.optBoolean(ANONYMIZE, builder.isAnonymized()));


            // get endpoint url
            builder.collectorEndpoint(opt(object, TRACKER_ENDPOINT_URL,
                    builder.getCollectorEndpoint()));

            // sample rate
            builder.samplingRate(opt(object, SAMPLE_RATE, builder.getSampleRateValue()));

            // storage_max_items
            builder.maxBucketSize(opt(object, STORAGE_MAX_ITEMS, builder.getMaxBucketSize()));

            // god mode key
            builder.debugFlag(opt(object, DEBUG_FLAG, builder.getDebugFlag()));

            // screen record active
            builder.screenShotsEnabled(object.optBoolean(SCREEN_RECORD_ACTIVE,
                    builder.isScreenShotEnabled()));

            // black listed versions list
            builder.blackListedVersionsList(opt(object, BLACKLISTED_VERSIONS, builder
                    .getBlackListedVersionsList()));
        } catch (JSONException e) {
            LOGGER.w(e, "cannot parse JSon Object. Exception thrown: ");
        } finally {
            builder.inSamplingInterval(mInSampleValidator.isInSampleInterval(builder
                    .getSampleRateValue()));
        }

    }

    private int opt(@NonNull JSONObject object, @NonNull String key, int defaultValue) {
        final int value = object.optInt(key, defaultValue);
        return value >= 0 ? value : defaultValue;
    }

    private float opt(@NonNull JSONObject object, @NonNull String key, float defaultValue) {
        final float value = (float) object.optDouble(key, defaultValue);
        return value >= 0.0f ? value : defaultValue;
    }

    @NonNull
    private String opt(@NonNull JSONObject object, @NonNull String key, String defaultValue) {
        final String value = object.optString(key, defaultValue);
        return !Strings.isNullOrEmpty(value) ? value : defaultValue;
    }
}
