package com.contentsquare.android.internal;

import static android.content.Context.WINDOW_SERVICE;
import static android.os.Build.VERSION.SDK_INT;
import static android.os.Build.VERSION_CODES.JELLY_BEAN_MR1;
import static android.util.TypedValue.applyDimension;
import static com.contentsquare.android.internal.model.JsonProxy.BATCH_APP_NAME;
import static com.contentsquare.android.internal.model.JsonProxy.BATCH_SDK_FLAVOR;
import static com.contentsquare.android.internal.model.JsonProxy.BATCH_SDK_TYPE;
import static com.contentsquare.android.internal.model.data.ActionEvent.CONNECTIVITY_ERROR;
import static com.contentsquare.android.internal.model.data.ActionEvent.EDGE;
import static com.contentsquare.android.internal.model.data.ActionEvent.HSPA;
import static com.contentsquare.android.internal.model.data.ActionEvent.LTE;
import static com.contentsquare.android.internal.model.data.ActionEvent.OFFLINE;
import static com.contentsquare.android.internal.model.data.ActionEvent.ORIENTATION_LANDSCAPE;
import static com.contentsquare.android.internal.model.data.ActionEvent.ORIENTATION_PORTRAIT;
import static com.contentsquare.android.internal.model.data.ActionEvent.OrientationType;
import static com.contentsquare.android.internal.model.data.ActionEvent.WIFI;

import android.annotation.SuppressLint;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.BatteryManager;
import android.os.Build;
import android.support.annotation.CheckResult;
import android.support.annotation.IntDef;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.WindowManager;

import com.contentsquare.android.R;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.model.data.ActionEvent.ConnectionType;
import com.contentsquare.android.internal.util.MathUtil;
import com.contentsquare.android.internal.util.Strings;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

/**
 * The DeviceInfo is the single point where you can get all Device related information.
 * It collects device information on the target device, and makes them available for the framework.
 */
public class DeviceInfo {

    /**
     * Constant providing the height Key in {@link #getDeviceResolution()}.
     */
    public static final String HEIGHT = "h";

    /**
     * Constant describing the width Key in {@link #getDeviceResolution()}.
     */
    public static final String WIDTH = "w";

    /**
     * Label used for the sdk version.
     */
    public static final String LABEL_SDK_VERSION = "sv";

    /**
     * Label used for the build version.
     */
    public static final String LABEL_SDK_BUILD = "sb";

    /**
     * Label used for the app version.
     */
    public static final String LABEL_APP_VERSOIN = "av";

    /**
     * Label used for the app flavor.
     */
    public static final String LABEL_APP_FLAVOR = "af";

    /**
     * Network Int Constant for Phone.
     */
    static final int PHONE_INT = 4;
    /**
     * Network Int constant for Tablet.
     */
    static final int TABLET_INT = 5;
    private static final double DEFAULT_BATTERY_LEVEL = -1f;
    private static final float MAX_BATTERY_LEVEL = 100f;
    private static final String ANDROID_SDK_IDENTIFIER = "sdk-android";
    private static final String ANDROID_SDK_DEBUG = "debug";
    private static final String ANDROID_SDK_RELEASE = "release";
    private static final int BATTERY_FETCH_TIMEOUT = 1000 * 30;

    private final Logger mLogger = new Logger("DeviceInfo");
    private final int mStatusBarHeight;
    private final Application mApplication;
    private String mDeviceName;
    @DeviceIntType
    private int mDeviceIntType;
    private Map<String, Integer> mDeviceResolution;
    private String mDeviceOs;
    private String mApplicationName;
    private String mApplicationVersion;
    private String mApplicationFlavor;
    private String mUserLanguage;
    private String mUserTimezone;
    private boolean mDebugBuild = true;
    private String mCarrierName;
    private long mBatteryFetchTimestamp = 0;
    @NonNull
    private Intent mCachedBatteryIntent;

    /**
     * Creates a new Device info object. This is a long lived object, due to some listeners which
     * are registered with the system, so it's distributed through the SingletonHolder
     * since we want a single instance.
     * Can also be injected using the {@link javax.inject.Inject} Annotation.
     *
     * @param application the running instance of the application.
     */
    public DeviceInfo(@NonNull Application application) {
        mApplication = application;
        mDebugBuild = com.contentsquare.android.BuildConfig.DEBUG;
        fillDeviceProperties();
        Resources res = application.getResources();
        int resourceId = res.getIdentifier("status_bar_height", "dimen", "android");
        if (resourceId > 0) {
            mStatusBarHeight = res.getDimensionPixelSize(resourceId);
        } else {
            mStatusBarHeight = 0;
        }
        mCachedBatteryIntent = loadBatteryIntent();
        mBatteryFetchTimestamp = System.currentTimeMillis();
    }

    public Application getApplication() {
        return mApplication;
    }

    public String getDeviceName() {
        return mDeviceName;
    }

    @DeviceIntType
    public int getDeviceIntType() {
        return mDeviceIntType;
    }

    public Map<String, Integer> getDeviceResolution() {
        return mDeviceResolution;
    }

    /**
     * Constructs a Json object which represents the device resolution.
     *
     * @return {@link JSONObject} describing the resolution of this device.
     */
    @Nullable
    public JSONObject getDeviceResolutionJson() {

        JSONObject object = new JSONObject();
        try {
            object.put(WIDTH, mDeviceResolution.get(WIDTH));
            object.put(HEIGHT, mDeviceResolution.get(WIDTH));
        } catch (JSONException e) {
            mLogger.e(e, "Failed to process device resolution for bundle.");
        }

        return object;
    }

    public String getDeviceOs() {
        return mDeviceOs;
    }

    public String getCarrierName() {
        return mCarrierName;
    }

    /**
     * Returns current state of battery level.
     *
     * @return an int representing the current level, or {@link #DEFAULT_BATTERY_LEVEL} on error.
     */
    public int getBatteryLevelNow() {
        Intent batteryIntent = getCachedtBatteryIntent();
        if (batteryIntent == null) {
            return (int) DEFAULT_BATTERY_LEVEL;
        }
        int level = batteryIntent.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
        int scale = batteryIntent.getIntExtra(BatteryManager.EXTRA_SCALE, -1);

        // Error checking that probably isn't needed but I added just in case.
        if (level == -1 || scale == -1) {
            return (int) DEFAULT_BATTERY_LEVEL;
        }

        return (int) (((float) level / (float) scale) * MAX_BATTERY_LEVEL);
    }

    @NonNull
    private Intent getCachedtBatteryIntent() {
        long now = System.currentTimeMillis();
        if (now - mBatteryFetchTimestamp > BATTERY_FETCH_TIMEOUT) {
            mCachedBatteryIntent = loadBatteryIntent();
        }
        return mCachedBatteryIntent;
    }

    private Intent loadBatteryIntent() {
        IntentFilter batteryFilter = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);
        return mApplication.registerReceiver(
                // the null here indicates that we're not registering a receiver.
                // rather just getting the last sticky broadcast message
                null,
                batteryFilter);
    }

    public String getApplicationName() {
        return mApplicationName;
    }

    public String getApplicationVersion() {
        return mApplicationVersion;
    }

    private String getApplicationFlavor() {
        return mApplicationFlavor;
    }

    public String getUserLanguage() {
        return mUserLanguage;
    }

    public String getUserTimezone() {
        return mUserTimezone;
    }

    /**
     * Checks if the device we're running on is a tablet.
     *
     * @return true if it's a tablet.
     */
    boolean isTablet() {
        return mApplication.getResources().getBoolean(R.bool.isTablet);
    }

    /**
     * Checks if the device we're running on is in landscape mode.
     *
     * @return true if it's in landscape.
     */
    boolean isInLandscape() {
        return mApplication.getResources().getBoolean(R.bool.isLandscape);
    }

    public boolean isDebugBuild() {
        return mDebugBuild;
    }

    /**
     * Returns This SDK's version number.
     *
     * @return the version number of the SDK.
     */
    public String getSdkVersion() {
        return com.contentsquare.android.BuildConfig.VERSION_NAME;
    }

    /**
     * Returns this SDK's build numer.
     *
     * @return the build number
     */
    int getSdkBuild() {
        return com.contentsquare.android.BuildConfig.VERSION_CODE;
    }

    public int getStatusBarHeight() {
        return mStatusBarHeight;
    }

    /**
     * Runtime conversion from DeviceIndependant pixels (dp) to Pixels (px).
     *
     * @return an int representing the pixels for the provided dips (rounded).
     */
    int convertDpToPx(int dips) {
        Resources res = mApplication.getResources();
        return (int) applyDimension(TypedValue.COMPLEX_UNIT_DIP, dips, res.getDisplayMetrics());

    }

    /**
     * Utility method to Calculate in runtime the action bar height.
     * The need for this runtime calculation is that the action bar is not the same height as the
     * tools bar, but occupy the same layout, and sometimes it's not present, so we need a runtime
     * calc for this.
     *
     * @return the current size of the actionbar/toolbar.
     */
    public int calculateActionBarHeight() {
        int actionBarHeight = 0;
        TypedValue tv = new TypedValue();
        if (mApplication.getApplicationContext().getTheme().resolveAttribute(
                android.R.attr.actionBarSize, tv, true)) {
            actionBarHeight = TypedValue.complexToDimensionPixelSize(tv.data,
                    mApplication.getApplicationContext().getResources().getDisplayMetrics());
        }

        return actionBarHeight;
    }

    private void fillDeviceProperties() {
        mLogger.d("initiating the device info.");

        mDeviceName = Build.MANUFACTURER + ' ' + Build.MODEL;
        mLogger.d("DeviceName: %s", mDeviceName);


        //device type
        if (isTablet()) {
            mDeviceIntType = TABLET_INT;
        } else {
            mDeviceIntType = PHONE_INT;
        }
        mLogger.d("DeviceType: %s", mDeviceIntType);

        //display size
        DisplayMetrics metrics = new DisplayMetrics();
        WindowManager windowManager = (WindowManager) mApplication.getSystemService(WINDOW_SERVICE);
        if (windowManager != null) { /* System services may return nulls */
            if (SDK_INT >= JELLY_BEAN_MR1) {
                windowManager.getDefaultDisplay().getRealMetrics(metrics);
            } else {
                windowManager.getDefaultDisplay().getMetrics(metrics);
            }
            mDeviceResolution = new HashMap<>();
            mDeviceResolution.put(HEIGHT, metrics.heightPixels);
            mDeviceResolution.put(WIDTH, metrics.widthPixels);
            mLogger.d("DeviceWidth: %d", metrics.widthPixels);
            mLogger.d("DeviceHeight: %d", metrics.heightPixels);
        }

        //deviceOS
        mDeviceOs = Build.VERSION.RELEASE;
        mLogger.d("DeviceOS: %s", mDeviceOs);

        //applicationName
        PackageManager packageManager = mApplication.getPackageManager();
        ApplicationInfo appInfo = mApplication.getApplicationInfo();
        mApplicationName = String.valueOf(appInfo.loadLabel(packageManager));

        mApplicationFlavor = processApplicationFlavor(getApplication().getApplicationContext(),
                "BUILD_TYPE");
        if (Strings.isNullOrEmpty(mApplicationFlavor)) {
            mApplicationFlavor = processApplicationFlavor(getApplication().getApplicationContext(),
                    "FLAVOR");
        }

        mLogger.d("ApplicationName: %s", mApplicationName);

        //applicationVersion
        try {
            PackageInfo info = packageManager.getPackageInfo(mApplication.getPackageName(), 0);
            mApplicationVersion = String.valueOf(info.versionName);

        } catch (PackageManager.NameNotFoundException e) {
            mLogger.w(e, "failed to get the PackageInfo for %s", mApplication.getPackageName());
            mApplicationVersion = "Unknown";
        }
        mLogger.d("ApplicationVersion: %s", mApplicationVersion);

        //user Locale
        mUserLanguage = Locale.getDefault().getDisplayName();
        mLogger.d("UserLanguage: %s", mUserLanguage);


        //user timezone
        mUserTimezone = TimeZone.getDefault().getID();
        mLogger.d("UserTimezone: %s", mUserTimezone);

        //carrier name
        TelephonyManager manager = (TelephonyManager)
                mApplication.getSystemService(Context.TELEPHONY_SERVICE);
        if (manager != null) {
            mCarrierName = manager.getNetworkOperatorName();
        }
    }


    /**
     * Retrieves the application flavor (debug/release) from the build config of the enclosing app
     * implementing this SDK.
     * We have to use reflection as the app package is unknown at the compile of the SDK.
     *
     * @param context   a context to get the packagename from.
     * @param fieldName the filed name which we want to query.
     * @return an object (to be casted)
     */
    @SuppressWarnings("unchecked")
    @CheckResult
    @Nullable
    private <T> T processApplicationFlavor(Context context, String fieldName) {
        try {
            Class<?> clazz = Class.forName(context.getPackageName() + ".BuildConfig");
            Field field = clazz.getField(fieldName);
            return (T) field.get(null);
        } catch (ClassNotFoundException e) {
            mLogger.e("Failed to get app Flavor %s", e.getMessage());
        } catch (NoSuchFieldException e) {
            mLogger.e("Failed to get app Flavor %s", e.getMessage());
        } catch (IllegalAccessException e) {
            mLogger.e("Failed to get app Flavor %s", e.getMessage());
        }
        return null;
    }

    /**
     * Checks to see if the app has a permission granted.
     *
     * @param permission the String representing a permission.
     * @return true if the permission is requested and approved.
     */
    public boolean hasPermissionGranted(@NonNull String permission) {
        int res = getApplication().getApplicationContext().checkCallingOrSelfPermission(permission);
        return (res == PackageManager.PERMISSION_GRANTED);
    }

    /**
     * Returns the current connectivity type as described by {@link ConnectionType} type constant.
     *
     * @return int representing the type of connection.
     */
    @ConnectionType
    @SuppressLint("MissingPermission")
    public int getActiveConnectionType() {
        ConnectivityManager conManager = (ConnectivityManager)
                mApplication.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (conManager != null && hasPermissionGranted("android.permission.ACCESS_NETWORK_STATE")) {
            NetworkInfo networkInfo = conManager.getActiveNetworkInfo();
            if (networkInfo == null) {
                return CONNECTIVITY_ERROR;
            }
            if (!networkInfo.isConnectedOrConnecting()) {
                return OFFLINE;
            }
            if (networkInfo.getType() == ConnectivityManager.TYPE_WIFI) {
                return WIFI;
            } else {
                return determineDataConnectionType(networkInfo.getSubtype());
            }
        }
        return CONNECTIVITY_ERROR;
    }

    @ConnectionType
    private int determineDataConnectionType(final int networkSubtype) {
        switch (networkSubtype) {
            case TelephonyManager.NETWORK_TYPE_1xRTT: // ~ 50-100 kbps
            case TelephonyManager.NETWORK_TYPE_CDMA: // ~ 14-64 kbps
            case TelephonyManager.NETWORK_TYPE_EDGE: // ~ 50-100 kbps
            case TelephonyManager.NETWORK_TYPE_EVDO_0: // ~ 400-1000 kbps
            case TelephonyManager.NETWORK_TYPE_EVDO_A: // ~ 600-1400 kbps
            case TelephonyManager.NETWORK_TYPE_GPRS: // ~ 100 kbps
                return EDGE;
            case TelephonyManager.NETWORK_TYPE_HSDPA: // ~ 2-14 Mbps
            case TelephonyManager.NETWORK_TYPE_HSPA: // ~ 700-1700 kbps
            case TelephonyManager.NETWORK_TYPE_HSUPA: // ~ 1-23 Mbps
            case TelephonyManager.NETWORK_TYPE_UMTS: // ~ 400-7000 kbps
            case TelephonyManager.NETWORK_TYPE_EHRPD: // ~ 1-2 Mbps
            case TelephonyManager.NETWORK_TYPE_EVDO_B: // ~ 5 Mbps
                return HSPA;
            case TelephonyManager.NETWORK_TYPE_HSPAP: // ~ 10-20 Mbps
            case TelephonyManager.NETWORK_TYPE_IDEN: // ~25 kbps
            case TelephonyManager.NETWORK_TYPE_LTE: // ~ 10+ Mbps
                return LTE;
            // Unknown
            case TelephonyManager.NETWORK_TYPE_UNKNOWN:
                return CONNECTIVITY_ERROR;
            default:
                return CONNECTIVITY_ERROR;
        }
    }

    /**
     * Returns a boolean representing the current charging state of the device (ac or usb)
     *
     * @return true if device is attached to any power source.
     */
    public boolean isDeviceCharging() {
        Intent batteryStatus = getCachedtBatteryIntent();
        boolean usbCharge;
        boolean acCharge;
        boolean wifiCharge = false;

        int chargePlug = batteryStatus.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1);
        usbCharge = chargePlug == BatteryManager.BATTERY_PLUGGED_USB;
        acCharge = chargePlug == BatteryManager.BATTERY_PLUGGED_AC;
        if (SDK_INT >= JELLY_BEAN_MR1) {
            wifiCharge = chargePlug == BatteryManager.BATTERY_PLUGGED_WIRELESS;
        }

        return usbCharge || acCharge || wifiCharge;
    }

    /**
     * Returns the current Screen orientation.
     */
    @OrientationType
    public int getScreenOrientation() {
        int orientation = getApplication().getResources().getConfiguration().orientation;
        if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
            return ORIENTATION_LANDSCAPE;
        } else {
            return ORIENTATION_PORTRAIT;
        }
    }

    /**
     * Creates a JSon object which is used as type origin according to our specs.
     *
     * @return JSon object representing bundle's origin
     */
    @SuppressFBWarnings(value = "DB_DUPLICATE_BRANCHES",
            justification = "Findbugs acting weirdly because we use the same value (when "
                    + "optimised in a different field")
    @NonNull
    public JSONObject getTypeOrigin() {
        JSONObject jsonOrigin = new JSONObject();
        try {
            jsonOrigin.put(BATCH_APP_NAME, getApplicationName());
            jsonOrigin.put(BATCH_SDK_TYPE, ANDROID_SDK_IDENTIFIER);
            jsonOrigin.put(BATCH_SDK_FLAVOR, mDebugBuild ? ANDROID_SDK_DEBUG : ANDROID_SDK_RELEASE);
        } catch (JSONException e) {
            mLogger.e(e, "Failed to get Type Origin json for event.");
        }
        return jsonOrigin;
    }

    /**
     * Creates a JSon object which is used as Version origin according to our specs.
     *
     * @return JSon object representing event's version origin
     */
    @NonNull
    public JSONObject getVersionOrigin() {
        JSONObject jsonOrigin = new JSONObject();
        try {
            jsonOrigin.put(LABEL_SDK_VERSION, getSdkVersion());
            jsonOrigin.put(LABEL_SDK_BUILD, getSdkBuild());
            jsonOrigin.put(LABEL_APP_VERSOIN, getApplicationVersion());
            jsonOrigin.put(LABEL_APP_FLAVOR, getApplicationFlavor());
        } catch (JSONException e) {
            mLogger.e(e, "Failed to get json version Origin for event.");
        }
        return jsonOrigin;
    }

    /**
     * Computes and returns the current heap memory for this process in KB.
     *
     * @param forProcess as the {@link Runtime} for which we want to process the available memory.
     * @return the current heap memory level in KB.
     */
    public int getCurrentAvailableFreeHeapMemory(Runtime forProcess) {
        final long currentUsedMemory = forProcess.totalMemory() - forProcess.freeMemory();
        final long currentAvailableMemory = forProcess.maxMemory() - currentUsedMemory;
        final long toKbDivider = 1024L;
        final long currentAvailableMemoryInKb = currentAvailableMemory / toKbDivider;
        return MathUtil.safeLongToInt(currentAvailableMemoryInKb);
    }

    /**
     * Device type int annotation.
     */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef( {PHONE_INT, TABLET_INT})
    public @interface DeviceIntType {
    }


}
