package com.contentsquare.android.internal;

import android.app.Activity;
import android.app.Application;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.async.ConfigRetrieverTask;
import com.contentsquare.android.internal.async.ConfigRetrieverTask.ConfigProviderAnswer;
import com.contentsquare.android.internal.config.GodModeConfiguration;
import com.contentsquare.android.internal.config.ProjectConfiguration;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.dagger.session.SessionComponent;
import com.contentsquare.android.internal.events.processing.EventsProcessor;
import com.contentsquare.android.internal.factories.TasksFactory;
import com.contentsquare.android.internal.listeners.CsActivityCallbacks;
import com.contentsquare.android.internal.listeners.CsComponentChange;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.model.EventsFactory;
import com.contentsquare.android.internal.model.JsonProxy;
import com.contentsquare.android.internal.model.data.ActionEvent;
import com.contentsquare.android.internal.model.data.AppExitEvent.AppExitEventBuilder;
import com.contentsquare.android.internal.model.data.AppHideEvent.AppHideEventBuilder;
import com.contentsquare.android.internal.preferences.PrefsHelper;
import com.contentsquare.android.internal.ui.glasspane.IGlassPane;
import com.contentsquare.android.internal.util.AssetManager;
import com.contentsquare.android.internal.util.ConfigurationHelper;
import com.contentsquare.android.internal.util.HttpConnection;
import com.contentsquare.android.internal.util.Strings;

import java.util.UUID;

import hugo.weaving.DebugLog;

/**
 * The Session class represents a User session within the scope of a single run of the app.
 * There is only a single session which represents a user working
 */
public class Session {
    /**
     * Label used for storing the sessionID preference.
     */
    private static final String SESSIONID = "sid";

    // the projectId with which the session was started
    @NonNull
    private final String mProjectId;
    // this session's main Event Processor which will handle all events during this session.
    @NonNull
    private final EventsProcessor mEventsProcessor;
    @NonNull
    private final Application mApplication;
    @NonNull
    private final AssetManager mAssets;
    @NonNull
    private final ConfigurationCompositor mCompositor;
    @NonNull
    private final DeviceInfo mDeviceInfo;
    @NonNull
    private final IGlassPane mGlassPane;
    private final TasksFactory mTasksFactory;
    private final EventsFactory mEventsFactory;
    private final JsonProxy mJsonProxy;
    // a session is valid ONLY if the network is reachable and connected
    @VisibleForTesting
    boolean mSessionIsValid = false;
    @NonNull
    @VisibleForTesting
    PrefsHelper mPreferences;
    @Nullable
    @VisibleForTesting
    CsActivityCallbacks mActivityListener = null;
    @NonNull
    @VisibleForTesting
    ConfigurationHelper mHelper;
    private int mScreenCount = 0;
    // the config for this session
    @NonNull
    private ProjectConfiguration mProjectConfig = ProjectConfiguration.builder().build();
    // the god mode config for this session
    @NonNull
    private GodModeConfiguration mGodModeConfiguration = new GodModeConfiguration();
    // an autoincrement int which represents this unique session (like a database id for a row)
    private int mSessionId;
    // the unique user's id which is specific to this user@device
    private String mUserId = null;
    private CsComponentChange mComponentListener = null;
    private final Logger mLog = new Logger("Session");
    private ConfigRetrieverTask mConfigRetrieverTask;

    /**
     * Constructs a Session object. This is the entrypoint of the start of the lifecycle of
     * a session.
     *
     * @param application     the active app's application
     * @param projectId       the projectID of the client received from the app.
     * @param prefsHelper     the preferences helper to be used by the session.
     * @param glassPane       the glasspane managing all touch events
     * @param eventsProcessor a non null events processor
     * @param assets          a non null asset manager
     * @param deviceInfo      this run's device info
     * @param compositor      a non null configuration compositor
     * @param tasksFactory    a non null tasks factory
     * @param eventsFactory   a factory for generating events
     * @param jsonProxy       a json proxy.
     */
    @SuppressWarnings("squid:S00107")
    public Session(
            @NonNull Application application,
            @NonNull String projectId,
            @NonNull PrefsHelper prefsHelper,
            @NonNull IGlassPane glassPane,
            @NonNull EventsProcessor eventsProcessor,
            @NonNull AssetManager assets,
            @NonNull DeviceInfo deviceInfo,
            @NonNull ConfigurationCompositor compositor,
            @NonNull TasksFactory tasksFactory,
            @NonNull EventsFactory eventsFactory,
            @NonNull JsonProxy jsonProxy) {
        mProjectId = projectId;
        mPreferences = prefsHelper;
        mEventsProcessor = eventsProcessor;
        mAssets = assets;
        mCompositor = compositor;
        mGlassPane = glassPane;
        mApplication = application;
        mDeviceInfo = deviceInfo;
        mTasksFactory = tasksFactory;
        mEventsFactory = eventsFactory;
        mJsonProxy = jsonProxy;
        mHelper = new ConfigurationHelper();
    }

    public int getSessionId() {
        return mSessionId;
    }

    @Nullable
    public String getUserId() {
        return mUserId;
    }

    @NonNull
    public EventsFactory getEventsFactory() {
        return mEventsFactory;
    }

    @NonNull
    public JsonProxy getJsonProxy() {
        return mJsonProxy;
    }

    /**
     * Checks both if the screenshots are enabled in {@link ProjectConfiguration} and also if god
     * mode is on in order to enabled the screen shots feature for the current session.
     *
     * @return true if the current Session can take Screenshots.
     */
    public boolean canTakeScreenShots() {
        return mProjectConfig.isScreenShotEnabled() && mGodModeConfiguration.isGodModeEnabled();
    }

    /**
     * Loads a user id, from preferences. Calls for {@link #generateAndStoreUserId()} if needed.
     */
    @VisibleForTesting
    void loadUserId() {
        mUserId = mHelper.getLastUserIdConfig();
        if (Strings.isNullOrEmpty(mUserId)) {
            mUserId = generateAndStoreUserId();
        }
    }

    /**
     * Generates and stores a {@link UUID#randomUUID()}.
     *
     * @return the value which was stored.
     */
    private String generateAndStoreUserId() {
        String uid = UUID.randomUUID().toString();
        mHelper.setUserIdConfig(uid);
        return uid;
    }

    /**
     * Loads the session by loading previous one, incrementing and storing the value.
     */
    @VisibleForTesting
    void loadSessionId() {
        int previousSession = mPreferences.getInt(SESSIONID, -1);
        mSessionId = previousSession + 1;

        //store this session's id for the next run.
        mPreferences.putInt(SESSIONID, mSessionId);

    }

    /**
     * Session Validation is checked against the server.
     * A session is considered invalid if the project ID is not retrievable from the server.
     * A session is considered valid AFTER the server validation occurs.
     *
     * @return true if the session was validated. False otherwise.
     */
    boolean isSessionValid() {
        return mSessionIsValid;
    }

    /**
     * Getter for the active RunConfiguration.
     *
     * @return - the existing and used RunConfiguration.
     */
    @NonNull
    public ProjectConfiguration getRunConfiguration() {
        return mProjectConfig;
    }

    /**
     * Chaining setter for a {@link ProjectConfiguration} which returns a Session.
     *
     * @param runConfiguration - an existing non null RunConfiguration.
     * @return the existing session.
     */
    @NonNull
    Session setRunConfiguration(@NonNull ProjectConfiguration runConfiguration) {
        mProjectConfig = runConfiguration;
        return this;
    }

    /**
     * Starts a Session.
     * This method will initiated session by checking whether there is any previous configuration
     * saved in sharedPrefs if not it will use the local config (complete config file) and proceed
     * to download the common base file and then the project config file.
     * This method also checks whether the application is in GOD MODE and gets its configuration.
     */
    @DebugLog
    public void startSession() {
        mLog.i("startSession Called");
        // loading the session id in the session and incrementing the counters in prefs.
        loadSessionId();
        // setting it in the events processor.
        getEventsProcessor().setSessionId(getSessionId());
        loadUserId();
        // unsubscribe from old provider if by any chance the stopSession was not called
        disposeConfigRetrieverIfExists();
        // subscribe to new provider
        retrieveConfig();
    }


    private void disposeConfigRetrieverIfExists() {
        if (mConfigRetrieverTask != null) {
            mConfigRetrieverTask.cancel(true);
        }
    }

    private void retrieveConfig() {
        SessionComponent sessionComponent = SingletonProvider.getSessionComponent();
        final HttpConnection httpConnection = sessionComponent
                .getHttpConnection();
        mConfigRetrieverTask = mTasksFactory.produceConfigRetrieverTask(
                new ConfigRetrieverTask.ConfigProviderCallback() {
                    @Override
                    public void processAnswer(@NonNull ConfigProviderAnswer answer) {
                        if (answer.isValid()) {
                            // we mark this session as valid if at least one config was provided
                            // TODO: 10/16/17 this might be an issue in the future for refactoring.
                            // this warning is "a false negative" but might be mis-interpreted in
                            // the future.
                            mProjectConfig = answer.projectConfiguration();
                            mGodModeConfiguration = answer.godModeConfiguration();
                            startListeners();
                            boolean inGodmode = mGodModeConfiguration.isGodModeEnabled();
                            boolean godOverride = mGodModeConfiguration.isLoggingEnabled();
                            if (inGodmode) {
                                Logger.overrideLogging(godOverride);
                            }
                            getEventsProcessor().setBucketSize(
                                    getRunConfiguration().getMaxBucketSize());
                        }
                    }
                },
                httpConnection,
                mAssets,
                mCompositor,
                mHelper);
        mConfigRetrieverTask.execute(getProjectId());
    }

    /**
     * Tells whether or not the God Mode was activated for this Session.
     *
     * @return true if the God Mode was activated for this Session
     */
    public boolean isGodModeOn() {
        return mGodModeConfiguration.isGodModeEnabled();
    }

    /**
     * Returns the GodMode config for the app.
     *
     * @return the active god mode config
     */
    @NonNull
    public GodModeConfiguration getGodModeConfig() {
        return mGodModeConfiguration;
    }

    @VisibleForTesting
    boolean hasConfigurationSaved() {
        return !Strings.isNullOrEmpty(mHelper.getLastClientConfig());
    }

    /**
     * Attaches all application listeners.
     */
    @VisibleForTesting
    void startListeners() {
        if (shouldTrackSession()) {
            mSessionIsValid = true;
            if (mActivityListener == null && mComponentListener == null) {
                mLog.i("the session was validated, attaching listeners");
                mActivityListener = new CsActivityCallbacks();
                mComponentListener = new CsComponentChange();
                mApplication.registerActivityLifecycleCallbacks(mActivityListener);
                mApplication.registerComponentCallbacks(mComponentListener);
            } else {
                mLog.i("the session is already active, moving along");
            }
        } else {
            mLog.i("the session Is INVALID, aborting");
            stopSession();
        }
    }

    @VisibleForTesting
    boolean shouldTrackSession() {
        ProjectConfiguration conf = getRunConfiguration();
        boolean track = conf.isTrackingEnabled() && conf.isInSampleInterval();
        mLog.d("shouldTrackSession : mSessionIsValid:%b ", mSessionIsValid);
        mLog.d("shouldTrackSession : isTrackerEngaged:%b ", conf.isTrackingEnabled());
        mLog.d("shouldTrackSession : isInAudience:%b ", conf.isInSampleInterval());
        mLog.d("shouldTrackSession : Final:%b ", track);
        return track;
    }

    /**
     * Stops a session which is already active.
     */
    public void stopSession() {
        mLog.i("stopSession Called");
        if (mSessionIsValid) {
            mLog.i("Stopping all listeners");
            mGlassPane.stopSession();

            // send hide and exit events right now.
            AppHideEventBuilder hideBuilder = getEventsFactory().produceEvent(ActionEvent.APP_HIDE);
            AppExitEventBuilder exitBuilder = getEventsFactory().produceEvent(ActionEvent.APP_EXIT);
            getEventsProcessor().processEvent(getJsonProxy().serializeToJson(hideBuilder.build()));
            getEventsProcessor().processEvent(getJsonProxy().serializeToJson(exitBuilder.build()));

            // flush now.
            getEventsProcessor().flushPendingEvents();
            // detach listeners
            mApplication.unregisterActivityLifecycleCallbacks(mActivityListener);
            mApplication.unregisterComponentCallbacks(mComponentListener);
            if (mActivityListener != null) {
                detachGlassPane(mActivityListener.getLiveActivity());
            }
            mActivityListener = null;
            mComponentListener = null;
            mSessionIsValid = false;
            disposeConfigRetrieverIfExists();
        } else {
            mLog.i("the session was INVALID, no listeners was registered.");
        }
    }


    @NonNull
    public EventsProcessor getEventsProcessor() {
        return mEventsProcessor;
    }

    @NonNull
    public IGlassPane getGlass() {
        return mGlassPane;
    }

    @NonNull
    public String getProjectId() {
        return mProjectId;
    }

    @Nullable
    public CsActivityCallbacks getActivityListener() {
        return mActivityListener;
    }

    /**
     * Attaches the Glass pane to the current activity.
     *
     * @param activity current activity
     */
    public void attachGlassPane(@NonNull Activity activity) {
        mGlassPane.attachGlassPane(activity);
        incrementScreenCount();
    }

    /**
     * Detaches the glass pane from the current activity.
     *
     * @param activity the activity which may or may not contain a glass pane
     */
    public void detachGlassPane(Activity activity) {
        mGlassPane.detachGlassPane(activity);
    }


    /**
     * Conveys a pause call to the glass listener and pauses the SDK.
     */
    public void pauseSession() {
        mGlassPane.pauseListeners();
    }

    /**
     * Conveys an un-pause call to the glass listener and resumes the work of the SDK.
     */
    public void resumeSession() {
        mGlassPane.resumeListeners();
    }

    /**
     * Increments the current screen count by 1.
     */
    private void incrementScreenCount() {
        mScreenCount++;
    }

    public int getScreenCount() {
        return mScreenCount;
    }

    /**
     * Produces a builder for type.
     *
     * @param eventAction the type of event.
     * @param <T>         a child of {@link ActionEvent}
     * @return the builder.
     */
    @SuppressWarnings("unchecked")
    public <T extends ActionEvent.Builder> T produceEventBuilder(
            @ActionEvent.EventAction int eventAction) {
        return getEventsFactory().produceEvent(eventAction);
    }
}
