package com.contentsquare.android.internal.async;

import static java.net.HttpURLConnection.HTTP_NOT_FOUND;
import static java.net.HttpURLConnection.HTTP_OK;

import android.os.AsyncTask;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.contentsquare.android.internal.ConfigurationCompositor;
import com.contentsquare.android.internal.config.GodModeConfiguration;
import com.contentsquare.android.internal.config.ProjectConfiguration;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.AssetManager;
import com.contentsquare.android.internal.util.ConfigurationHelper;
import com.contentsquare.android.internal.util.HttpConnection;
import com.contentsquare.android.internal.util.Strings;
import com.contentsquare.android.internal.util.UriBuilder;
import com.contentsquare.android.internal.validator.GodModeClipboardValidator;
import com.contentsquare.android.internal.validator.GodModeConfigValidator;


/**
 * This is an AsyncTask which will be fetching, processing and providing through a callback an
 * instance of the {@link com.contentsquare.android.internal.config.ProjectConfiguration} to the
 * interested party.
 * <p>
 * Please note that the callback will always be called on the UI Thread.
 * Please also note that this class encapsulates also the logic through which we check if we are
 * currently in GOD MODE, fetches the GOD MODE configuration in this case and returns this to the
 * waiting party.
 */

public class ConfigRetrieverTask extends AsyncTask<String, Void, ConfigRetrieverTask
        .ConfigProviderAnswer> {

    /**
     * TAG to be used in the logger for this class.
     */
    public static final String TAG = "ConfigRetrieverTask";

    @NonNull
    private final ConfigProviderCallback mCallback;
    @NonNull
    private final HttpConnection mHttpConnection;
    @NonNull
    private final AssetManager mAssetsManager;
    @NonNull
    private final ConfigurationCompositor mConfigurationCompositor;
    @NonNull
    private final ConfigurationHelper mHelper;
    @NonNull
    private final GodModeClipboardValidator mGodModeClipboardValidator;
    @NonNull
    private final GodModeConfigValidator mGodModeConfigValidator;
    @NonNull
    private final Logger mLogger;
    @NonNull
    private final UriBuilder mUriBuilder;

    private ProjectConfiguration mDefaultConfig;

    /**
     * Produces a new instance of {@link ConfigRetrieverTask}.
     *
     * @param callback                  as {@link ConfigProviderCallback}
     * @param httpConnection            as {@link HttpConnection}
     * @param assetManager              as {@link AssetManager}
     * @param configurationCompositor   as {@link ConfigurationCompositor}
     * @param helper                    as {@link ConfigurationHelper}
     * @param uriBuilder                as {@link UriBuilder}
     * @param godModeClipboardValidator as {@link GodModeClipboardValidator}
     * @param godModeConfigValidator    as {@link GodModeConfigValidator}
     * @param logger                    as {@link Logger}
     */
    public ConfigRetrieverTask(@NonNull ConfigProviderCallback callback,
                               @NonNull HttpConnection httpConnection,
                               @NonNull AssetManager assetManager,
                               @NonNull ConfigurationCompositor configurationCompositor,
                               @NonNull ConfigurationHelper helper,
                               @NonNull UriBuilder uriBuilder,
                               @NonNull GodModeClipboardValidator godModeClipboardValidator,
                               @NonNull GodModeConfigValidator godModeConfigValidator,
                               @NonNull Logger logger) {

        this.mCallback = callback;
        this.mHttpConnection = httpConnection;
        this.mAssetsManager = assetManager;
        this.mConfigurationCompositor = configurationCompositor;
        this.mHelper = helper;
        this.mUriBuilder = uriBuilder;
        this.mGodModeClipboardValidator = godModeClipboardValidator;
        this.mGodModeConfigValidator = godModeConfigValidator;
        this.mLogger = logger;
    }

    @Override
    protected void onPreExecute() {
        super.onPreExecute();
        mDefaultConfig = fetchDefaultConfig();
        // Refresh to get the latest value from the clipboard
        mGodModeClipboardValidator.fetchClipboardData();
        onPostExecute(new ConfigProviderAnswer(mDefaultConfig, new GodModeConfiguration()));
    }

    @Override
    protected ConfigProviderAnswer doInBackground(String... params) {
        final String projectId = params[0];
        final String projectConfigUrl;
        GodModeConfiguration godModeConfiguration = godModeConfiguration();
        final boolean godModeValid = mGodModeConfigValidator
                .isGodModeValidFor(projectId, godModeConfiguration);

        // check is god mode is valid
        if (!godModeValid) {
            // the god mode was not valid, go back to default disabled one
            godModeConfiguration = new GodModeConfiguration();
        }


        // check god mode details and switch the ProjectConfiguration Endpoint
        final boolean shouldUseDebugConfig = godModeValid && godModeConfiguration
                .shouldUseDebugConfig();
        if (shouldUseDebugConfig) {
            projectConfigUrl = mUriBuilder.buildProjectConfigUrlForDebug(projectId);
        } else {
            projectConfigUrl = mUriBuilder.buildProjectConfigUrl(projectId);
        }

        final String jsonString = fetchConfig(projectConfigUrl);

        if (jsonString == null) {
            // fail here
            return ConfigProviderAnswer.invalid();
        }

        // save the newly fetched config into the SharedPrefs
        final ProjectConfiguration projectConfiguration = mConfigurationCompositor
                .updateRunConfigurationFromConfig(mDefaultConfig, jsonString);
        mHelper.setLastConfig(projectConfiguration.toJson().toString());
        return new ConfigProviderAnswer(projectConfiguration, godModeConfiguration);

    }

    @Override
    protected void onPostExecute(ConfigProviderAnswer answer) {
        super.onPostExecute(answer);
        if (isCancelled()) {
            return;
        }
        if (answer != null) {
            mCallback.processAnswer(answer);
        }
    }

    @NonNull
    private GodModeConfiguration godModeConfiguration() {
        if (mGodModeClipboardValidator.isGodModeKeyVerified(mDefaultConfig.getGodModeKey())) {
            final String godModeConfig = fetchConfig(
                    mUriBuilder.godModeConfigEndpoint());
            return GodModeConfiguration.load(godModeConfig);
        }
        return GodModeConfiguration.load(null);
    }

    @Nullable
    private String fetchConfig(@NonNull String path) {

        HttpConnection.HttpResponse projectResult =
                mHttpConnection.performGetWithFormdata(path, null);
        if (projectResult.getStatus() == HTTP_OK) {
            mLogger.d("Got HTTP_OK for path: [%s]", path);
            String stringResponse = projectResult.getStringResponse();
            if (!TextUtils.isEmpty(stringResponse)) {
                return stringResponse;
            }
        }


        if (projectResult.getStatus() == HTTP_NOT_FOUND) {
            mLogger.e("Got HTTP_NOT_FOUND for path [%s]", path);
        }
        if (projectResult.getStatus() > HTTP_OK) {
            mLogger.e("Got HTTP Status: [%d] for path: [%s]",
                    projectResult.getStatus(),
                    path);
        }

        final String errorMsg = "Invalid ProjectId provided. "
                + "Please verify the project ID you provided when "
                + "making the call to the ContentSquareSDK. \n ContentSquare "
                + "Exiting...";
        //CHECKSTYLE:OFF
        android.util.Log.e("ContentSquare",
                errorMsg);
        //CHECKSTYLE:ON
        return null;

    }

    @NonNull
    private ProjectConfiguration fetchDefaultConfig() {

        final String jsonConfig;
        if (hasCachedConfig()) {
            mLogger.d("has configuration saved");
            jsonConfig = mHelper.getLastClientConfig();
            // adding last known config from server, locally cached with a ttl
        } else {
            mLogger.d("No configuration saved - Loading Base Config");
            jsonConfig = mAssetsManager.loadBaseConfigFromAsset();
        }

        return mConfigurationCompositor.getRunConfiguration(jsonConfig);
    }

    private boolean hasCachedConfig() {
        return !Strings.isNullOrEmpty(mHelper.getLastClientConfig());
    }

    /**
     * The callback through which this AsyncTask communicates with the interested party.
     */
    public interface ConfigProviderCallback {
        /**
         * This method can be called multiple times for the same request. It can return in the
         * first place the default config (either the previously stored on or the base one) and
         * in the end it can return the newly fetched configuration from server side.
         * <p>
         * Note: This method is Only called once per answer or a total of twice per async task
         * lifecycle. Once from {@link AsyncTask#onPreExecute()} and once from {@link
         * AsyncTask#doInBackground(Object[])}.
         *
         * @param answer as the {@link ConfigProviderAnswer}
         */
        void processAnswer(@NonNull ConfigProviderAnswer answer);
    }

    /**
     * The answer that this task will return by calling the
     * {@link ConfigProviderCallback#processAnswer(ConfigProviderAnswer)}.
     */
    public static class ConfigProviderAnswer {
        @Nullable
        private final ProjectConfiguration mProjectConfiguration;
        @Nullable
        private final GodModeConfiguration mGodModeConfiguration;

        private final boolean mIsValid;

        ConfigProviderAnswer(@NonNull ProjectConfiguration projectConfiguration,
                             @Nullable GodModeConfiguration godModeConfiguration) {
            this(projectConfiguration, godModeConfiguration, true);
        }

        ConfigProviderAnswer(@Nullable ProjectConfiguration projectConfiguration,
                             @Nullable GodModeConfiguration godModeConfiguration,
                             boolean isValid) {
            this.mProjectConfiguration = projectConfiguration;
            this.mGodModeConfiguration = godModeConfiguration;
            this.mIsValid = isValid;
        }

        @NonNull
        static ConfigProviderAnswer invalid() {
            return new ConfigProviderAnswer(null, null, false);
        }

        /**
         * Provides the new instance of the {@link GodModeConfiguration}
         *
         * @return the newly parsed instance retrieved from server side of
         * {@link GodModeConfiguration}.
         */
        @Nullable
        public GodModeConfiguration godModeConfiguration() {
            return mGodModeConfiguration;
        }

        /**
         * This can contain either the newly parsed instance of the {@link ProjectConfiguration}
         * by updating the current stored default config with the received json config retrieved
         * from server side or the default stored config.
         *
         * @return the projectConfiguration as {@link ProjectConfiguration}
         */
        @Nullable
        public ProjectConfiguration projectConfiguration() {
            return mProjectConfiguration;
        }


        /**
         * Tells if this answer is valid or not.
         *
         * @return true if the answer is valid and contains any valid data.
         */
        public boolean isValid() {
            return mIsValid;
        }
    }

}
