package com.contentsquare.android.internal.async;

import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.logging.Logger;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * A wrapper around a {@link ThreadPoolExecutor} which is to be used for all async executions in
 * the SDK.
 */
public class ThreadExecutor {

    private static final int CORE_POOL_SIZE = 1;
    private static final int AVAILABLE_PROCESSORS = Runtime.getRuntime().availableProcessors();
    private static final long KEEP_ALIVE = 30L;
    private static final TimeUnit TIME_UNIT = TimeUnit.SECONDS;
    private static final int BLOCKING_QUEUE_SIZE = 10;
    @VisibleForTesting
    final BlockingQueue<Runnable> mWorkQueue;
    @VisibleForTesting
    ThreadPoolExecutor mExecutor;
    private static final Logger LOGGER = new Logger("ThreadExecutor");

    /**
     * Constructs a thread executor which is a wrapper around {@link ThreadPoolExecutor}.
     * The instance has a min pool size of 1, max pool size = number of cores.
     * keep alive is 30s.
     */
    public ThreadExecutor() {
        this(new ArrayBlockingQueue<Runnable>(BLOCKING_QUEUE_SIZE),
                new RejectedExecutionHandler() {
                    @Override
                    public void rejectedExecution(final Runnable runnable,
                            final ThreadPoolExecutor executor) {
                        if (runnable instanceof TransactionRunnable) {
                            LOGGER.i("Transaction Rejected! rolling back.");
                            ((TransactionRunnable) runnable).rollBack();
                        } else {
                            LOGGER.e("Rejected! Cannot rollback! Not an instance of "
                                    + "TransactionRunnable");
                        }
                    }
                });

    }

    /**
     * Constructor from provided work queue and handler objects.
     * @param workQueue       the proposed work queue.
     * @param rejectedHandler the rejection handler.
     */
    @VisibleForTesting
    ThreadExecutor(final BlockingQueue<Runnable> workQueue,
            final RejectedExecutionHandler rejectedHandler) {
        LOGGER.i("Building a thredpool executor maxSize %d", AVAILABLE_PROCESSORS);
        mExecutor = new ThreadPoolExecutor(
                CORE_POOL_SIZE,
                AVAILABLE_PROCESSORS,
                KEEP_ALIVE,
                TIME_UNIT,
                workQueue,
                rejectedHandler);
        mWorkQueue = workQueue;
        mExecutor.allowCoreThreadTimeOut(false);

    }

    /**
     * Dispatch a runnable to the ThreadPool. This method will try to add the runnable to the queue
     * and if successfully dispatched, it will return true.
     * @param runnable the runnable to be executed
     * @return true if the dispatch was successful, false otherwise.
     */
    public boolean dispatch(@Nullable final Runnable runnable) {

        if (runnable == null) {
            return false;
        }

        mExecutor.execute(runnable);
        LOGGER.i("dispatching, mWorkQueue.size(%d) ", mWorkQueue.size());
        LOGGER.w("pool status: "
                        + "ActiveCount[%d], "
                        + "QueueSize[%d], "
                        + "TaskCount[%d], "
                        + "PoolSize[%d], "
                        + "isShutdown[%b], "
                        + "isTerminated[%b], "
                        + "isTerminating[%b], ",
                mExecutor.getActiveCount(),
                mExecutor.getQueue().size(),
                mExecutor.getTaskCount(),
                mExecutor.getPoolSize(),
                mExecutor.isShutdown(),
                mExecutor.isTerminated(),
                mExecutor.isTerminating()
        );
        return true;
    }

    /**
     * Clears the existing worker queue for the thread pool.
     * This method ONLY clears the queue, it knows nothing on how to handle any transaction related
     * logic.
     */
    public void clearQueue() {
        mWorkQueue.clear();
    }
}
