package com.contentsquare.android.internal.config;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.Strings;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * The GodModeConfiguration is the runtime configuration that turns our SDK into GOD MODE.
 * This configuration will be fetched from the internet whenever the phone is connected to the CS
 * network.
 */
public class GodModeConfiguration {
    private static final String GOD_MODE = "god_mode";
    private static final String USE_DEBUG_CONF = "use_debug_conf";
    private static final String ENABLE_LOG = "enable_log";
    private static final String SD_STORAGE_EVENTS = "sd_storage_events";
    private static final String SD_STORAGE_SCREENS = "sd_storage_screens";
    private static final String UPLOAD_URL = "network_storage";
    private static final String PRINT_TOUCHPATH = "print_touchpath";
    private static final String SAVE_SCREENSHOT_LOCALLY = "save_screenshot_locally";
    private static final String WHITELIST = "whitelist";

    @SuppressWarnings("squid:S1450")
    private static final Logger LOG = new Logger("GodModeConfiguration");

    private boolean mGodModeEnabled = false;
    private boolean mUseDebugConfig = false;
    private boolean mDebugLogsEnabled = false;
    private boolean mUseSdForEventStorage = false;
    private boolean mUseSdForScreenStorage = false;
    private String mScreengraphUploadUrl = null;
    private boolean mLogTouchPaths = false;
    private boolean mStoreScreengraphsLocally = false;
    private String mWhiteList;

    // TODO: 10/3/17 Move this into JsonProxy. Get rid of the static access.

    /**
     * This method converts the configuration godDirectives from JSON to an object.
     *
     * @param godDirectives String of the configuration file
     * @return GodModeConfiguration object
     */
    @NonNull
    @SuppressWarnings("squid:S1166")
    public static GodModeConfiguration load(@Nullable String godDirectives) {

        GodModeConfiguration godConfig = new GodModeConfiguration();
        godConfig.setGodModeOn(false);

        if (Strings.isNullOrEmpty(godDirectives)) {
            //exit early in the case of explicit null, to skip un-necessary execution.
            return godConfig;
        }

        JSONObject object;
        try {
            object = new JSONObject(godDirectives);

            if (object.has(USE_DEBUG_CONF)) {
                boolean debugConf = getBooleanField(object, USE_DEBUG_CONF);
                godConfig.setUseDebugConfig(debugConf);
            }

            if (object.has(ENABLE_LOG)) {
                boolean enableLog = getBooleanField(object, ENABLE_LOG);
                godConfig.setLogEnabled(enableLog);
            }

            if (object.has(SD_STORAGE_EVENTS)) {
                boolean storeEvents = getBooleanField(object, SD_STORAGE_EVENTS);
                godConfig.setStoreEventsOnSdCard(storeEvents);
            }

            if (object.has(SD_STORAGE_SCREENS)) {
                boolean storeScreen = getBooleanField(object, SD_STORAGE_SCREENS);
                godConfig.setStoreScreensOnSdCard(storeScreen);
            }

            if (object.has(UPLOAD_URL)) {
                String url = getStringField(object, UPLOAD_URL);
                godConfig.setScreengraphUploadUrl(url);
            }

            if (object.has(PRINT_TOUCHPATH)) {
                boolean touchPath = getBooleanField(object, PRINT_TOUCHPATH);
                godConfig.setPrintTouchPad(touchPath);
            }

            if (object.has(SAVE_SCREENSHOT_LOCALLY)) {
                boolean saveScreenShot = getBooleanField(object, SAVE_SCREENSHOT_LOCALLY);
                godConfig.setStoreScreenshotsLocally(saveScreenShot);
            }

            godConfig.setWhiteList(getStringField(object, WHITELIST));

            if (object.has(GOD_MODE)) {
                boolean godMode = getBooleanField(object, GOD_MODE);
                godConfig.setGodModeOn(godMode);
            }

        } catch (NullPointerException | JSONException e) {
            LOG.e("Error: %s", e.getMessage());
            LOG.w("God mode is disabled due to an error in the config.");
        }

        return godConfig;
    }

    private static boolean getBooleanField(@NonNull JSONObject json, String label) {
        return json.optBoolean(label, false);
    }

    @NonNull
    private static String getStringField(@NonNull JSONObject json, String label) {
        return json.optString(label, null);
    }

    // these are setters, and due to naming, checkstyle fails to detect getters
    //CHECKSTYLE:OFF
    public boolean isGodModeEnabled() {
        return mGodModeEnabled;
    }

    @VisibleForTesting
    private void setGodModeOn(boolean godMode) {
        mGodModeEnabled = godMode;
    }

    public boolean shouldUseDebugConfig() {
        return mUseDebugConfig;
    }

    @VisibleForTesting
    private void setUseDebugConfig(boolean useDebugConfig) {
        mUseDebugConfig = useDebugConfig;
    }

    public boolean isLoggingEnabled() {
        return mDebugLogsEnabled;
    }

    @VisibleForTesting
    private void setLogEnabled(boolean enableLog) {
        mDebugLogsEnabled = enableLog;
    }

    boolean storeEventsOnSdCard() {
        return mUseSdForEventStorage;
    }

    @VisibleForTesting
    private void setStoreEventsOnSdCard(boolean sdStorageEvents) {
        mUseSdForEventStorage = sdStorageEvents;
    }

    boolean storeScreensOnSdCard() {
        return mUseSdForScreenStorage;
    }

    @VisibleForTesting
    private void setStoreScreensOnSdCard(boolean sdStorageScreens) {
        mUseSdForScreenStorage = sdStorageScreens;
    }

    public String getScreengraphUploadUrl() {
        return mScreengraphUploadUrl;
    }

    @VisibleForTesting
    private void setScreengraphUploadUrl(String uploadUrl) {
        mScreengraphUploadUrl = uploadUrl;
    }

    boolean shouldPrintTouchPath() {
        return mLogTouchPaths;
    }

    @VisibleForTesting
    private void setPrintTouchPad(boolean printTouchPath) {
        mLogTouchPaths = printTouchPath;
    }

    boolean storeScreenshotsLocally() {
        return mStoreScreengraphsLocally;
    }

    @VisibleForTesting
    private void setStoreScreenshotsLocally(boolean saveScreenShotLocally) {
        mStoreScreengraphsLocally = saveScreenShotLocally;
    }

    @Nullable
    public String getWhiteList() {
        return mWhiteList;
    }

    private void setWhiteList(String whiteList) {
        this.mWhiteList = whiteList;
    }

    //CHECKSTYLE:ON
}
