package com.contentsquare.android.internal.config;

import android.support.annotation.NonNull;

import com.contentsquare.android.internal.ConfigurationCompositor;
import com.contentsquare.android.internal.logging.Logger;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * A ProjectConfiguration represents a user's currently saved configuration of the SDK or the
 * default one in case of first usage of the application or in an expired ttl for the saved one.
 * The ProjectConfiguration describes all properties related to this single run, but is subject to
 * updates. For this reason, the active runtime configuration should *ALWAYS* be obtained from
 * {@link com.contentsquare.android.internal.Session}.
 * A ProjectConfiguration is composed of the following properties.
 * <ul>
 * <li> <i>csProjectId</i> the ContentSquare client database allocated project
 * id as primary key {@link Integer} > 0 </li>
 * <li> <i>version</i> as the version of this config {@link Integer} > 0 </li>
 * <li> <i>enabled</i> as {@link Boolean} which enables/disables the tracker</li>
 * <li> <i>screenshots</i> as {@link Boolean} which tells if screenshots are enabled or not</li>
 * <li> <i>anonymize</i> as {@link Boolean} which enables/disables anonymous mode</li>
 * <li> <i>endpoint</i> as {@link String} as the collector endpoint
 * including the PROTOCOL and PORT</li>
 * <li> <i>sample</i> as {@link Float} 0 < sample < 1  for usage sampling </li>
 * <li> <i>ssid</i> as {@link String} the God Mode Wifi SSID </li>
 * <li> <i>bucket</i> as {@link Integer} size of transactions (max bucket size) </li>
 * <li> <i>blacklisted</i> as {@link String} comma separated list of versions which are considered
 * <li> <i>inSampleInterval</i>  as {@link Boolean} tells if this device is or not in the
 * accepted sample interval. In case of <code>false</code>it will stop tracking this session</li>
 * as 'disabled' </li>
 * </ul>
 */

public class ProjectConfiguration {

    // the project id as primary key
    private final int mCsProjectId;

    // the version of this config
    private final int mConfigVersion;

    // enable/disable the tracker feature
    private final boolean mTrackerEnabled;

    // enable/disable screen shots feature
    private final boolean mScreenShots;

    // enable/disable anonymous mode
    private final boolean mAnonymize;

    // the collector endpoint (e.g. protocol://host:port/path/to/service)
    private final String mEndpoint;

    // the sampling rate for tracking data (0 <= sampleRate <= 1)
    private final float mSampleRate;

    // the GOD MODE trigger key
    private final String mDebugFlag;

    // the size of the current bucket of events (getMaxBucketSize)
    private final int mBucketSize;

    // deprecated versions separated by comma (e.g. 1.1.3,2.3,4.0)
    private final String mBlackListedAppVersions;

    // whether this device is in the accepted sampling interval
    private final boolean mInSampleInterval;

    private final Logger mLogger = new Logger(ProjectConfiguration.class.getSimpleName());

    private ProjectConfiguration(@NonNull Builder builder) {
        mCsProjectId = builder.mCsProjectId;
        mConfigVersion = builder.mVersion;
        mTrackerEnabled = builder.mEnabled;
        mScreenShots = builder.mScreenShots;
        mAnonymize = builder.mAnonymize;
        mEndpoint = builder.mEndpoint;
        mSampleRate = builder.mSampleRate;
        mDebugFlag = builder.mDebugFlag;
        mBucketSize = builder.mMaxBucketSize;
        mBlackListedAppVersions = builder.mBlackListed;
        mInSampleInterval = builder.mInSampleInterval;
    }

    /**
     * Creates a new instance of {@link Builder}.
     *
     * @return a new {@link Builder}
     */
    @NonNull
    public static Builder builder() {
        return new Builder();
    }

    public int getCsProjectId() {
        return mCsProjectId;
    }

    public int getConfigVersion() {
        return mConfigVersion;
    }

    public boolean isTrackingEnabled() {
        return mTrackerEnabled;
    }

    public boolean isScreenShotEnabled() {
        return mScreenShots;
    }

    public boolean isAnonymousEnabled() {
        return mAnonymize;
    }

    @NonNull
    public String getCollectorEndpoint() {
        return mEndpoint;
    }

    public float getTrackingSampleRate() {
        return mSampleRate;
    }

    @NonNull
    public String getGodModeKey() {
        return mDebugFlag;
    }

    public int getMaxBucketSize() {
        return mBucketSize;
    }

    @NonNull
    public String getBlackListedVersionsList() {
        return mBlackListedAppVersions;
    }

    /**
     * Transforms and returns this object as a {@link JSONObject}.
     *
     * @return this object as {@link JSONObject}
     */
    // TODO: 9/28/17 Move this method into the Proxy which Tancho
    @NonNull
    public JSONObject toJson() {
        final JSONObject object = new JSONObject();
        try {
            object.put(ConfigurationCompositor.CS_PROJECT_ID, mCsProjectId);
            object.put(ConfigurationCompositor.CONFIG_VERSION, mConfigVersion);
            object.put(ConfigurationCompositor.BLACKLISTED_VERSIONS, mBlackListedAppVersions);
            object.put(ConfigurationCompositor.SCREEN_RECORD_ACTIVE, mScreenShots);
            object.put(ConfigurationCompositor.TRACKER_ACTIVE, mTrackerEnabled);
            object.put(ConfigurationCompositor.ANONYMIZE, mAnonymize);
            object.put(ConfigurationCompositor.DEBUG_FLAG, mDebugFlag);
            object.put(ConfigurationCompositor.SAMPLE_RATE, mSampleRate);
            object.put(ConfigurationCompositor.STORAGE_MAX_ITEMS, mBucketSize);
            object.put(ConfigurationCompositor.TRACKER_ENDPOINT_URL, mEndpoint);
        } catch (JSONException e) {
            mLogger.e(e, "JSONException while trying to serialize the current instance into a "
                    + "JSONObject");
        }
        return object;
    }

    public boolean isInSampleInterval() {
        return mInSampleInterval;
    }

    public static class Builder {
        private int mCsProjectId;
        private int mVersion;
        private boolean mEnabled;
        private boolean mScreenShots;
        private boolean mAnonymize;
        private String mEndpoint = "";
        private float mSampleRate;
        private String mDebugFlag = "";
        private int mMaxBucketSize;
        private String mBlackListed = "";
        private boolean mInSampleInterval;


        private Builder() {
        }

        /**
         * @param anonymize to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder anonymize(boolean anonymize) {
            this.mAnonymize = anonymize;
            return this;
        }

        /**
         * @param projectId to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder csProjectId(int projectId) {
            mCsProjectId = projectId;
            return this;
        }

        /**
         * @param configVersion to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder configVersion(int configVersion) {
            mVersion = configVersion;
            return this;
        }

        /**
         * @param trackingEnabled to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder trackingEnabled(boolean trackingEnabled) {
            mEnabled = trackingEnabled;
            return this;
        }

        /**
         * @param screenShotsEnabled to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder screenShotsEnabled(boolean screenShotsEnabled) {
            mScreenShots = screenShotsEnabled;
            return this;
        }

        /**
         * @param collectorEndpoint to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder collectorEndpoint(@NonNull String collectorEndpoint) {
            mEndpoint = collectorEndpoint;
            return this;
        }

        /**
         * @param samplingRate to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder samplingRate(float samplingRate) {
            mSampleRate = samplingRate;
            return this;
        }

        /**
         * @param debugFlag to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder debugFlag(@NonNull String debugFlag) {
            mDebugFlag = debugFlag;
            return this;
        }

        /**
         * @param maxBucketSize to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder maxBucketSize(int maxBucketSize) {
            mMaxBucketSize = maxBucketSize;
            return this;
        }

        /**
         * @param list to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder blackListedVersionsList(@NonNull String list) {
            mBlackListed = list;
            return this;
        }

        /**
         * @param inSampleInterval to set.
         * @return this as {@link Builder}
         */
        @NonNull
        public Builder inSamplingInterval(boolean inSampleInterval) {
            mInSampleInterval = inSampleInterval;
            return this;
        }

        public boolean isAnonymized() {
            return mAnonymize;
        }

        public boolean isTrackerEnabled() {
            return mEnabled;
        }

        public boolean isScreenShotEnabled() {
            return mScreenShots;
        }

        public float getSampleRateValue() {
            return mSampleRate;
        }

        public int getCsProjectId() {
            return mCsProjectId;
        }

        public int getVersion() {
            return mVersion;
        }

        public int getMaxBucketSize() {
            return mMaxBucketSize;
        }

        public String getCollectorEndpoint() {
            return mEndpoint;
        }

        public String getDebugFlag() {
            return mDebugFlag;
        }

        public String getBlackListedVersionsList() {
            return mBlackListed;
        }

        /**
         * @return an immutable instance of {@link ProjectConfiguration}.
         */
        @NonNull
        public ProjectConfiguration build() {
            return new ProjectConfiguration(this);
        }
    }
}
