package com.contentsquare.android.internal.dagger;

import android.app.Application;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.dagger.application.ApplicationComponent;
import com.contentsquare.android.internal.dagger.application.ApplicationModule;
import com.contentsquare.android.internal.dagger.application.DaggerApplicationComponent;
import com.contentsquare.android.internal.dagger.session.DaggerSessionComponent;
import com.contentsquare.android.internal.dagger.session.SessionComponent;
import com.contentsquare.android.internal.dagger.session.SessionModule;
import com.contentsquare.android.internal.dagger.session.TrackingModule;
import com.contentsquare.android.internal.logging.Logger;

/**
 * Singleton holder for all Dagger related modules.
 * this class is made Abstract so that it's methods are considered as static and cannot be
 * instantiated.
 */

public class SingletonProvider {

    private static ApplicationComponent sAppComponent;
    private static SessionComponent sSessionComponent;

    @VisibleForTesting
    SingletonProvider() {
        //to assure that this class is not instantiated
        throw new UnsupportedOperationException("This class cannot be instantiated.");
    }

    /**
     * Initializes the Dagger singleton provider.
     *
     * @param context   a context instance
     * @param projectId the project id (lowercase alphanumeric string).
     */
    public static boolean init(@NonNull Context context, @NonNull String projectId) {
        if (sAppComponent == null && sSessionComponent == null) {
            Logger.init();
            Application application = (Application) context.getApplicationContext();
            if (sAppComponent == null) {
                sAppComponent = DaggerApplicationComponent.builder()
                        .applicationModule(new ApplicationModule(application))
                        .build();
            }
            if (sSessionComponent == null) {
                sSessionComponent = DaggerSessionComponent.builder()
                        .applicationComponent(getAppComponent())
                        .sessionModule(new SessionModule(projectId))
                        .trackingModule(new TrackingModule())
                        .build();
            }
            return true;
        } else {
            // init was called before no need to start a parallel session.
            return false;
        }
    }

    /**
     * Retrieves the session component which is holding singletons related to the session.
     *
     * @return the SessionComponent
     */
    @NonNull
    public static SessionComponent getSessionComponent() {
        return sSessionComponent;
    }

    /**
     * Sets the dagger Session component to the instance provided (useful for mocking).
     * This method is not to be used in the app, as it's mostly meant for unit tests.
     *
     * @param sessionComponent the session component
     */
    public static void setSessionComponent(SessionComponent sessionComponent) {
        sSessionComponent = sessionComponent;
    }

    /**
     * Retrieves the Application component which is holding singletons related to the Application.
     *
     * @return the ApplicationComponent
     */
    @NonNull
    public static ApplicationComponent getAppComponent() {
        return sAppComponent;
    }

    /**
     * Sets the dagger app component to the instance provided (useful for mocking).
     * This method is not to be used in the app, as it's mostly meant for unit tests.
     *
     * @param appComponent the app component
     */
    public static void setAppComponent(ApplicationComponent appComponent) {
        sAppComponent = appComponent;
    }

    /**
     * Resets the singleton provider.
     */
    public static void reset() {
        sAppComponent = null;
        sSessionComponent = null;
    }
}
