package com.contentsquare.android.internal.dagger.application;

import android.app.Application;
import android.support.annotation.NonNull;

import com.contentsquare.android.internal.ConfigurationCompositor;
import com.contentsquare.android.internal.DeviceInfo;
import com.contentsquare.android.internal.factories.TasksFactory;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.preferences.PrefsHelper;
import com.contentsquare.android.internal.util.AssetManager;
import com.contentsquare.android.internal.util.DateTimeUtil;
import com.contentsquare.android.internal.util.UriBuilder;
import com.contentsquare.android.internal.util.ViewUtil;
import com.contentsquare.android.internal.validator.InSampleIntervalValidator;

import javax.inject.Singleton;

import dagger.Module;
import dagger.Provides;

/**
 * Dagger module producing any singletons required on the application level.
 */

@Module
public class ApplicationModule {

    private final Application mApplication;

    /**
     * Constructs an {@link ApplicationModule} containing the running application instance and a
     * preferences helper.
     *
     * @param application the instance of this application.
     */
    public ApplicationModule(@NonNull Application application) {
        mApplication = application;
    }

    /**
     * Provides the {@link Application} instance.
     *
     * @return the application instance
     */
    @Provides
    @Singleton
    @NonNull
    public Application provideApplication() {
        return mApplication;
    }

    /**
     * Provides a {@link PrefsHelper} instance.
     *
     * @return the {@link PrefsHelper} instance
     */
    @Provides
    @Singleton
    @NonNull
    public PrefsHelper providePreferencesHelper() {
        return new PrefsHelper(mApplication.getBaseContext());
    }

    /**
     * Provides a {@link ConfigurationCompositor} instance.
     *
     * @return the {@link ConfigurationCompositor} instance
     */
    @Provides
    @Singleton
    @NonNull
    public ConfigurationCompositor provideCompositor(@NonNull PrefsHelper prefsHelper) {
        return new ConfigurationCompositor(new InSampleIntervalValidator(prefsHelper,
                new Logger(InSampleIntervalValidator.TAG)));
    }

    /**
     * Provides a {@link DeviceInfo} instance.
     *
     * @return the {@link DeviceInfo} instance
     */
    @Provides
    @Singleton
    @NonNull
    public DeviceInfo provideDeviceInfo() {
        return new DeviceInfo(mApplication);
    }

    /**
     * Provides a {@link AssetManager } instance.
     *
     * @return the {@link AssetManager} instance
     */
    @Provides
    @Singleton
    @NonNull
    public AssetManager provideAssetManager() {
        return new AssetManager(mApplication.getBaseContext());
    }

    /**
     * Provides a {@link TasksFactory} instance.
     *
     * @return the {@link TasksFactory} instance
     */
    @Provides
    @Singleton
    @NonNull
    public TasksFactory providesTasksFactoryInstance(@NonNull UriBuilder uriBuilder,
                                                     @NonNull Application application) {
        return new TasksFactory(uriBuilder, application.getBaseContext());
    }

    /**
     * Provide a {@link UriBuilder} instance.
     *
     * @param deviceInfo as {@link DeviceInfo}
     * @return the {@link UriBuilder} instance
     */
    @Provides
    @Singleton
    @NonNull
    public UriBuilder provideUriBuilder(DeviceInfo deviceInfo) {
        return new UriBuilder(deviceInfo);
    }

    /**
     * Provide a {@link ViewUtil} instance.
     *
     * @return the {@link ViewUtil} instance
     */
    @Provides
    @Singleton
    @NonNull
    public ViewUtil provideViewUtil() {
        return new ViewUtil();
    }

    /**
     * Provide a {@link DateTimeUtil} instance.
     *
     * @return the {@link DateTimeUtil} instance
     */
    @Provides
    @Singleton
    @NonNull
    public DateTimeUtil provideDateTimeUtil() {
        return new DateTimeUtil();
    }
}
