package com.contentsquare.android.internal.dagger.session;

import android.app.Application;
import android.support.annotation.NonNull;

import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.dagger.UserScope;
import com.contentsquare.android.internal.dagger.application.ApplicationComponent;
import com.contentsquare.android.internal.events.processing.EventsProcessor;
import com.contentsquare.android.internal.events.processing.NetworkTransaction;
import com.contentsquare.android.internal.listeners.CsActivityCallbacks;
import com.contentsquare.android.internal.listeners.CsComponentChange;
import com.contentsquare.android.internal.model.JsonProxy;
import com.contentsquare.android.internal.screengraph.PathGenerator;
import com.contentsquare.android.internal.screengraph.TreeTraverser;
import com.contentsquare.android.internal.screenmonitoring.IScreenMonitoringService;
import com.contentsquare.android.internal.screenmonitoring.ScreenMonitoringModule;
import com.contentsquare.android.internal.ui.processor.descriptors.PathDescriptor;
import com.contentsquare.android.internal.util.HttpConnection;
import com.contentsquare.android.internal.util.ScreenRecorder;

import dagger.Component;

/**
 * SessionComponent for the Dagger Injected singleton instances.
 */
@UserScope
@Component(
        dependencies = ApplicationComponent.class,
        modules = {SessionModule.class,
                HttpModule.class,
                TrackingModule.class,
                AsyncModule.class,
                ScreenMonitoringModule.class,
                SessionUtilitiesModule.class})
public interface SessionComponent {
    /**
     * Returns the active {@link Session} corresponding with this run managed by Dagger.
     * The session is  the main connector between the system and the api.
     *
     * @return - the active (open) session.
     */
    Session getSession();

    /**
     * Get the {@link HttpConnection} object owned by this session, managed by gradle.
     *
     * @return the
     */
    HttpConnection getHttpConnection();

    /**
     * Returns an instance of {@link PathGenerator}.
     *
     * @return the PathGenerator
     */
    PathGenerator getPathGenerator();


    /**
     * Returns the active application.
     *
     * @return an applicatin.
     */
    Application getApplication();

    /**
     * Returns an instance of {@link ScreenRecorder}.
     *
     * @return the ScreenRecorder
     */
    ScreenRecorder getScreenRecorder();


    /**
     * Returns an instance of {@link PathDescriptor}.
     *
     * @return the {@link PathDescriptor}
     */
    PathDescriptor getPathDescriptor();

    /**
     * Returns an instance of {@link IScreenMonitoringService} as a singleton in the scope of this
     * component.
     *
     * @return an {@link IScreenMonitoringService}
     */
    IScreenMonitoringService getScreenMonitoringService();

    /**
     * Returns an instance of {@link EventsProcessor} as a singleton in the scope of this component.
     * @return a {@link EventsProcessor}
     */
    EventsProcessor getEventsProcessor();

    /**
     * Returns an instance of a {@link JsonProxy} as a singleton in the scope of this component.
     * @return a {@link JsonProxy}
     */
    JsonProxy getJsonProxy();

    /**
     * Injector for CsActivityCallbacks objects.
     *
     * @param csActivityCallbacks a non null validator
     */
    void inject(@NonNull CsActivityCallbacks csActivityCallbacks);

    /**
     * Injector for CsComponentChange objects.
     *
     * @param csComponentChange a non null validator
     */
    void inject(@NonNull CsComponentChange csComponentChange);

    /**
     * Injector for event.processing.NetworkTransaction objects.
     *
     * @param networkTransaction a non null transaction
     */
    void inject(@NonNull NetworkTransaction networkTransaction);

    /**
     * Injector for network.NetworkTransaction objects.
     *
     * @param networkTransaction a non null transaction
     */
    void inject(@NonNull com.contentsquare.android.internal.network.NetworkTransaction
                        networkTransaction);

    /**
     * Injector for {@see TreeTraverser} objects.
     *
     * @param treeTraverser a non null {@see TreeTraverser}.
     */
    void inject(@NonNull TreeTraverser treeTraverser);


    /**
     * Injector for {@see PathGenerator} objects.
     *
     * @param pathGenerator a non null {@see PathGenerator}.
     */
    void inject(@NonNull PathGenerator pathGenerator);
}
