package com.contentsquare.android.internal.dagger.session;

import android.app.Application;
import android.support.annotation.NonNull;

import com.contentsquare.android.internal.ConfigurationCompositor;
import com.contentsquare.android.internal.DeviceInfo;
import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.dagger.UserScope;
import com.contentsquare.android.internal.events.processing.EventsProcessor;
import com.contentsquare.android.internal.factories.TasksFactory;
import com.contentsquare.android.internal.model.EventsFactory;
import com.contentsquare.android.internal.model.JsonProxy;
import com.contentsquare.android.internal.preferences.PrefsHelper;
import com.contentsquare.android.internal.screengraph.PathGenerator;
import com.contentsquare.android.internal.ui.glasspane.IGlassPane;
import com.contentsquare.android.internal.util.AssetManager;
import com.contentsquare.android.internal.util.ScreenRecorder;
import com.contentsquare.android.internal.util.UriBuilder;
import com.contentsquare.android.internal.util.ViewUtil;

import dagger.Module;
import dagger.Provides;

/**
 * Dagger Singleton Module providing the running instance of
 * {@link Session}.
 */
@Module
public class SessionModule {

    private final String mProjectId;

    /**
     * Constructor for the {@link Session} provider module.
     *
     * @param projectId the client app package name.
     */
    public SessionModule(@NonNull String projectId) {
        mProjectId = projectId;
    }

    /**
     * Module provider which provides, the active {@link Session}.
     *
     * @param application     the running app instance.
     * @param prefsHelper     the preferences helper for this session.
     * @param glassPane       the glass pane for this session
     * @param eventsProcessor processor for this session
     * @param assets          the asset manager for this session
     * @param deviceInfo      the device info for this session
     * @param compositor      the configuration compositor
     * @param tasksFactory    the AsyncTasks factory
     * @return the Active instance's Session (valid or invalid)
     */
    @Provides
    @UserScope
    @NonNull
    @SuppressWarnings("squid:S00107")
    Session provideSession(
            @NonNull Application application,
            @NonNull PrefsHelper prefsHelper,
            @NonNull IGlassPane glassPane,
            @NonNull EventsProcessor eventsProcessor,
            @NonNull AssetManager assets,
            @NonNull DeviceInfo deviceInfo,
            @NonNull ConfigurationCompositor compositor,
            @NonNull TasksFactory tasksFactory,
            @NonNull EventsFactory eventsFactory,
            @NonNull JsonProxy jsonProxy
    ) {
        return new Session(application, mProjectId, prefsHelper, glassPane, eventsProcessor, assets,
                deviceInfo, compositor, tasksFactory, eventsFactory, jsonProxy);

    }

    /**
     * Provide a {@link PathGenerator} instance.
     *
     * @param application a link to the existing {@link Application} instance.
     * @param viewUtil    a link to the existing {@link ViewUtil} instance.
     * @return the {@link PathGenerator} instance
     */
    @Provides
    @UserScope
    @NonNull
    public PathGenerator providePathGenerator(@NonNull Application application,
                                              @NonNull ViewUtil viewUtil) {
        return new PathGenerator(application, viewUtil);
    }

    /**
     * Provide a {@link ScreenRecorder} instance.
     *
     * @param deviceInfo    a link to the existing {@link DeviceInfo} instance.
     * @param session       a link to the existing {@link Session} instance.
     * @param uriBuilder    a link to the existing {@link UriBuilder} instance.
     * @param viewUtil      a link to the existing {@link ViewUtil} instance.
     * @return the {@link ScreenRecorder} instance
     */
    @Provides
    @UserScope
    @NonNull
    public ScreenRecorder provideScreenRecorder(
            @NonNull DeviceInfo deviceInfo,
            @NonNull Session session,
            @NonNull UriBuilder uriBuilder,
            @NonNull ViewUtil viewUtil) {
        return new ScreenRecorder(deviceInfo, session, uriBuilder, viewUtil);
    }


}

