package com.contentsquare.android.internal.events.processing;

import android.content.Context;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;
import android.support.v4.util.Pair;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.FileStorageUtil;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * Extracting variables from EventsProcessor.
 */
public class EventStorageManager {


    private final FileStorageUtil mStorageUtil;
    private final Logger mLogger = new Logger("EventStorageManager");
    @VisibleForTesting
    int mFilename = 0;
    @VisibleForTesting
    String mAppFilesLocation = null;
    @VisibleForTesting
    int mActiveBucketCurrentSize = 0;

    /**
     * Constructor For EventStorageManager.
     *
     * @param context Application context
     */
    public EventStorageManager(Context context) {
        mStorageUtil = new FileStorageUtil();
        mAppFilesLocation = context.getFilesDir().getAbsolutePath();
    }

    /**
     * Constructor for testing.
     *
     * @param storageUtil FileStorage Utility
     */
    public EventStorageManager(@NonNull final FileStorageUtil storageUtil) {
        mStorageUtil = storageUtil;
    }

    int getActiveBucketSize() {
        return mActiveBucketCurrentSize;
    }

    @VisibleForTesting
    int getUnsentBucketsSize(int sessionId) {
        return mStorageUtil.getNumberOfItemsInFolder(String.valueOf(sessionId));
    }

    void addToActiveBucket(@IntRange(from = 0) final int sessionId,
            @NonNull final JSONObject event) {
        String jsonString;
        jsonString = event.toString();
        sessionPrepare(sessionId);
        storeEventForSession(sessionId, jsonString);
    }

    int getActiveBucketId() {
        return mFilename;
    }

    /**
     * Prepares the session storage.
     * In the case of files, validate that the CS folder exists and that a folder with the session
     * id exists. Otherwise, create them.
     *
     * @param sessionId the id of the session.
     */
    private void sessionPrepare(final int sessionId) {
        boolean resetNeeded = mStorageUtil.prepareAndReset(sessionId, mAppFilesLocation);

        if (resetNeeded) {
            mFilename = 0;
            mActiveBucketCurrentSize = 0;
        }
        mStorageUtil.prepareFile(String.valueOf(sessionId), String.valueOf(mFilename));
    }

    /**
     * Appends this string to a file. We rely on the fact the file exists already and was created
     * when {@link #sessionPrepare(int)} was called.
     *
     * @param sessionId  the session id.
     * @param jsonString the string to append.
     */
    private void storeEventForSession(int sessionId, String jsonString) {
        mStorageUtil.appendStringToFile(String.valueOf(sessionId), String.valueOf(mFilename),
                jsonString);
        mActiveBucketCurrentSize++;
    }

    @VisibleForTesting
    List<JSONObject> getBucketContent(@IntRange(from = 0) final int sessionId,
            @IntRange(from = 0) final int activeBucketId) {

        List<String> lines = mStorageUtil.readFileContentByLine(String.valueOf(sessionId),
                String.valueOf(activeBucketId));
        List<JSONObject> result = new ArrayList<>(lines.size());
        for (String stringItem : lines) {
            try {
                result.add(new JSONObject(stringItem));
            } catch (JSONException e) {
                mLogger.e(e, "DATALOSS : Failed to serialize string to JSon Object");
            }
        }
        return result;
    }

    @VisibleForTesting
    void resetActiveBucket() {
        mActiveBucketCurrentSize = 0;
        mFilename++;
    }

    void deleteBucket(@IntRange(from = 0) int sessionId, @IntRange(from = 0) int filename) {
        mStorageUtil.deleteFile(String.valueOf(sessionId), String.valueOf(filename));
        mStorageUtil.cleanupFolder(String.valueOf(sessionId));
    }

    /**
     * Gets all the Usent Files.
     *
     * @return a {@link List} of {@link Pair} with its Session Number and bucket number.
     */
    @NonNull
    public List<Pair<Integer, Integer>> getAllUnsentData() {
        List<Pair<Integer, Integer>> result = new LinkedList<>();

        String[] sessionsLocations = listAllSessionsInStorage();

        for (String session : sessionsLocations) {
            File sessionFoler = new File(session);
            if (isValidSession(sessionFoler)) {
                mLogger.w("Unsent session: %s - with path %s", sessionFoler.getName(),
                        sessionFoler.getPath());
                int sessionId = Integer.parseInt(sessionFoler.getName());
                File[] batch = getBucketBatch(sessionFoler);
                for (File bucket : batch) {
                    mLogger.w("Unsent bucket: %s - with path %s", bucket.getName(),
                            bucket.getPath());
                    int bucketId = Integer.parseInt(bucket.getName());
                    result.add(new Pair<>(sessionId, bucketId));
                }
            } else {
                mLogger.e("Error : Cannot read folder! %s", sessionFoler.getAbsolutePath());
            }
        }
        return result;
    }

    @NonNull
    private String[] listAllSessionsInStorage() {
        return mStorageUtil.getMainDirectoryList();

    }

    private boolean isValidSession(File unsentSession) {
        return mStorageUtil.isDirectory(unsentSession) && mStorageUtil.readable(unsentSession);
    }

    private File[] getBucketBatch(File file) {
        return mStorageUtil.getFileList(file);
    }
}

