package com.contentsquare.android.internal.events.processing;

import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.support.v4.util.Pair;

import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.logging.Logger;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class holds the single responsibility of processing events from the listeners, sending them
 * to buckets, buffering them, and at the end sending them over the network.
 * It can define jobs if needed and bundle events to be sent.
 */
public class EventsProcessor {
    private static final int DEFAULT_SIZE = 100;
    private final Logger mLogger = new Logger("EventsProcessor");
    private final EventStorageManager mEventStorage;
    @VisibleForTesting
    int mMaxBucketSize = DEFAULT_SIZE;
    @VisibleForTesting
    Map<String, Pair<Integer, Integer>> mInProcessBuckets = new HashMap<>();
    private int mSessionId;

    /**
     * Constructs a Event processor with an empty bucket for data.
     */
    public EventsProcessor(@NonNull EventStorageManager eventStorage) {
        mEventStorage = eventStorage;
    }

    /**
     * Takes an event which can be nullable, if null, discard, otherwise dumps it in the bucket.
     * Once the bucket reaches {@link #mMaxBucketSize} it will start wrapping events with headers
     * and start sending them to the server.
     *
     * @param event the event we're processing.
     */
    public void processEvent(@Nullable JSONObject event) {
        mLogger.i("[new format] Event received Bucket size =%d  max size = %d",
                mEventStorage.getActiveBucketSize(),
                mMaxBucketSize);
        if (event == null) {
            mLogger.d("Event is null exiting");
            return;
        }
        mLogger.d("processing event: %s", event.toString());
        mEventStorage.addToActiveBucket(mSessionId, event);
        if (mEventStorage.getActiveBucketSize() >= mMaxBucketSize) {
            processMainBucket();
        }
    }

    /**
     * Processes the active bucket by creating a network transaction, executing it and creating a
     * new bucket.
     */
    @VisibleForTesting
    void processMainBucket() {
        mLogger.i(
                "MAIN bucket: bucket reached maxSize, processing events and creating a new bucket");
        boolean secondaryBatch = false;
        processBucket(mSessionId, mEventStorage.getActiveBucketId(), secondaryBatch);
        mEventStorage.resetActiveBucket();
    }

    /**
     * Flushes any event which is pending. This call is meant for either quickly ending a session or
     * handling {@link com.contentsquare.android.internal.listeners.CsComponentChange#onLowMemory()}
     * calls.
     */
    public void flushPendingEvents() {
        // premature processing of the main bucket will send all events and trigger
        // a send of any unsent events as well.
        processMainBucket();
    }

    /**
     * Processes the active bucket by creating a network transaction, executing it and creating a
     * new bucket.
     *
     * @param sessionId      the active session id
     * @param activeBucketId the active bucket id (filename inside the session)
     */
    @VisibleForTesting
    void processBucket(int sessionId, int activeBucketId, boolean processInBatch) {
        mLogger.w("processing received session %d, bucket of %d .", sessionId, activeBucketId);
        NetworkTransaction transaction = createTransaction(sessionId, activeBucketId,
                processInBatch);
        moveToInprocess(transaction, sessionId, activeBucketId);
        transaction.execute();
    }

    private void moveToInprocess(NetworkTransaction transaction, int sessionId,
                                 int activeBucketId) {
        mInProcessBuckets.put(transaction.getId(), new Pair<>(sessionId, activeBucketId));
    }

    @NonNull
    @VisibleForTesting
    NetworkTransaction createTransaction(@IntRange(from = 0) final int sessionId,
                                         @IntRange(from = 0) final int activeBucketId,
                                         boolean secondaryBatch) {
        List<JSONObject> bucket = mEventStorage.getBucketContent(sessionId, activeBucketId);
        Session session = SingletonProvider.getSessionComponent().getSession();
        return new NetworkTransaction(
                session,
                bucket,
                secondaryBatch,
                SingletonProvider.getAppComponent().getUriBuilder());
    }

    /**
     * Markes a requested {@link NetworkTransaction} as failed. This method will move the events
     * processed in this transaction to the list of unsent batches
     *
     * @param id the id of the transaction, provided by {@link NetworkTransaction#getId()}.
     */
    void failTransaction(String id) {
        mLogger.i("Received a failed transaction with id : %s, moving to unsent data", id);
        mInProcessBuckets.remove(id);
    }

    /**
     * Marks a {@link NetworkTransaction} as passed with success.
     * This means doing the proper cleanup of the data lists.
     *
     * @param id the id of the transaction.
     */
    void passTransaction(@NonNull String id) {
        processSentBuckets(id);

    }

    /**
     * Marks a {@link NetworkTransaction} as passed with success.
     * This means doing the proper cleanup of the data lists.
     * Also prepares to process the unset buckets.
     *
     * @param id the id of the transaction.
     */
    void passTransactionAndGotoNextBatch(@NonNull String id) {
        processSentBuckets(id);
        mLogger.i("starting to process unsent buckets, current bucket : %s", id);
        processUsentBuckets();

    }

    private void processUsentBuckets() {
        mLogger.i("processing unsent buckets");

        List<Pair<Integer, Integer>> batchesToProcess = mEventStorage.getAllUnsentData();

        for (Pair<Integer, Integer> pair : batchesToProcess) {
            int active = mEventStorage.getActiveBucketId();
            int session = mSessionId;
            if (pair.first == session && pair.second == active) {

                mLogger.i("Active buckets skipped session: %d - filename: %d ", session, active);
                //skip the currently active bucket
                continue;
            }
            boolean batch = true;
            processBucket(pair.first, pair.second, batch);
        }
    }

    /**
     * Sets the size of the bucket. Default is 100.
     *
     * @param storageMaxItems the new size
     */
    public void setBucketSize(int storageMaxItems) {
        mLogger.d("setting bucket size to %d", storageMaxItems);
        mMaxBucketSize = storageMaxItems;
    }

    int getMaxBucketSize() {
        return mMaxBucketSize;
    }

    @VisibleForTesting
    int getActiveBucketSize() {
        return mEventStorage.getActiveBucketSize();
    }

    public void setSessionId(@IntRange(from = 0) final int sessionId) {
        mSessionId = sessionId;
    }

    private void processSentBuckets(@NonNull String id) {
        mLogger.i("Passed transaction with id : %s, processing to unsent data", id);
        Pair<Integer, Integer> item = mInProcessBuckets.get(id);
        mInProcessBuckets.remove(id);

        // item can legally be null (happened once on showroom prive)
        if (item != null) {
            mEventStorage.deleteBucket(item.first, item.second);
        }
    }

}
