package com.contentsquare.android.internal.events.processing;

import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.async.ThreadExecutor;
import com.contentsquare.android.internal.async.TransactionRunnable;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.model.JsonProxy;
import com.contentsquare.android.internal.model.data.EventsBundle;
import com.contentsquare.android.internal.model.data.EventsBundle.BundleBuilder;
import com.contentsquare.android.internal.util.HttpConnection;
import com.contentsquare.android.internal.util.UriBuilder;

import org.json.JSONObject;

import java.util.List;
import java.util.UUID;

import javax.inject.Inject;

/**
 * A Network Transaction wraps the logic of transmitting a bucket of data from the {@link
 * EventsProcessor} to the server, including the logic around processing the status of the
 * transaction it's self. It's sole purpose in life is to handle data sent from the {@link
 * EventsProcessor} and finish their lifecycle one way or another.
 * If a network transaction fails, it will leave the data as is, otherwise it will delete it from
 * local storage as processed.
 * TODO: 9/17/17 this class needs to be refactored out. There is yet another Network Transaction,
 * which was written lately in network package, we should abstract parts of this one to use that.
 */
public class NetworkTransaction {
    private final Session mSession;
    private final String mId;
    @Inject
    ThreadExecutor mExecutor;
    @VisibleForTesting
    boolean mBatch = false;
    private final List<JSONObject> mBucket;
    private final Logger mLogger = new Logger("NetworkTransaction");
    @NonNull
    private final UriBuilder mUriBuilder;

    /**
     * Constructs a NetworkTransaction from a list of events.
     *
     * @param bucket      the list of events to be processed.
     * @param uriBuilder as the {@link UriBuilder}
     */
    NetworkTransaction(
            @NonNull Session session,
            @NonNull List<JSONObject> bucket,
            boolean isBatch,
            @NonNull UriBuilder uriBuilder) {
        this.mUriBuilder = uriBuilder;
        SingletonProvider.getSessionComponent().inject(this);
        mId = UUID.randomUUID().toString();
        mLogger.d("New Transaction created. Uid : %s", mId);
        mBucket = bucket;
        mBatch = isBatch;
        mSession = session;
    }

    /**
     * Retrieves the {@link #mId} of this transaction.
     *
     * @return a non null id.
     */
    public String getId() {
        return mId;
    }

    /**
     * Executes this transaction (attempts to make a network connection and send to the server).
     */
    public void execute() {
        scheduleSend();
    }

    @VisibleForTesting
    @SuppressWarnings("squid:S1188")
    void scheduleSend() {
        Runnable runnable = new TransactionRunnable() {
            @Override
            public void rollBack() {
                fail();
            }

            @Override
            public void run() {
                mLogger.i("Calling HTTP request for bucket of %d items", mBucket.size());
                JsonProxy proxy = mSession.getJsonProxy();
                JSONObject jsonData = proxy.serializeToJson(getBundle());
                if (jsonData == null) {
                    //if we received a null JSon we fail early.
                    fail();
                    return;
                }
                String endpoint = mSession.getRunConfiguration().getCollectorEndpoint();
                HttpConnection.HttpResponse result =
                        getHttpConnection().performPostWithJson(
                                mUriBuilder.buildEventsUrl(endpoint),
                                jsonData);


                if (result.positive()) {
                    pass();
                } else {
                    fail();
                }
            }
        };
        boolean dispatchSuccessful = mExecutor.dispatch(runnable);

        if (!dispatchSuccessful) {
            mLogger.w("dispatch not successful, failing!");
            fail();
        } else

        {
            mLogger.w("dispatch success, should exec soon!");
        }

    }

    @NonNull
    EventsBundle getBundle() {
        return new BundleBuilder().setPayload(mBucket).build();
    }

    @VisibleForTesting
    HttpConnection getHttpConnection() {
        return new HttpConnection();
    }

    @VisibleForTesting
    void fail() {
        mLogger.w("dispatching runnable was unscuccessful, failing this transaction.");
        mSession.getEventsProcessor().failTransaction(mId);
    }

    @VisibleForTesting
    void pass() {
        //process next batch if not secondary
        if (mBatch) {
            mLogger.w("SencondaryBatch Transaction passed, triggering cleanup in events processor");
            mSession.getEventsProcessor().passTransaction(mId);
        } else {
            mLogger.w("PrimaryBatch Transaction passed, triggering cleanup in events processor");
            mSession.getEventsProcessor().passTransactionAndGotoNextBatch(mId);
        }
    }


    /**
     * Returns the bucket which this {@link NetworkTransaction} is holding.
     *
     * @return the data.
     */
    public List<JSONObject> getData() {
        return mBucket;
    }
}
