package com.contentsquare.android.internal.listeners;

import static com.contentsquare.android.internal.model.data.ActionEvent.APP_EXIT;
import static com.contentsquare.android.internal.model.data.ActionEvent.APP_HIDE;
import static com.contentsquare.android.internal.model.data.ActionEvent.APP_SHOW;
import static com.contentsquare.android.internal.model.data.ActionEvent.APP_START;
import static com.contentsquare.android.internal.model.data.ActionEvent.PAGE_VIEW;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;
import android.support.v4.app.Fragment;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.FrameLayout;

import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.dagger.session.SessionComponent;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.model.JsonProxy;
import com.contentsquare.android.internal.model.data.AppExitEvent.AppExitEventBuilder;
import com.contentsquare.android.internal.model.data.AppHideEvent.AppHideEventBuilder;
import com.contentsquare.android.internal.model.data.AppShowEvent.AppShowEventBuilder;
import com.contentsquare.android.internal.model.data.AppStartEvent.AppStartEventBuilder;
import com.contentsquare.android.internal.model.data.PageViewEvent;
import com.contentsquare.android.internal.model.data.PageViewEvent.PageViewBuilder;
import com.contentsquare.android.internal.screengraph.PathGenerator;
import com.contentsquare.android.internal.screenmonitoring.IScreenChangedCallback;
import com.contentsquare.android.internal.screenmonitoring.IScreenMonitoringService;
import com.contentsquare.android.internal.util.FrameworkUtility;

import hugo.weaving.DebugLog;

/**
 * This is an implementation of the {@link Application.ActivityLifecycleCallbacks} interface. This
 * listeners is attached to the system as the main entry point for activity tracking and following
 * the activity stack.
 */
public class CsActivityCallbacks implements Application.ActivityLifecycleCallbacks {
    /**
     * Timeout for hiding an app. If an app is not on screen for more than 5s.
     * it's considered hidden.
     */
    private static final long HIDE_TIMEOUT = 5_000;
    /**
     * Timeout for exiting an app. If an app is not on screen for more than 60s.
     * it's considered exited.
     */
    private static final long EXIT_TIMEOUT = 60_000;
    @VisibleForTesting
    @NonNull
    final IScreenMonitoringService mScreenMonitoringService;
    @VisibleForTesting
    @NonNull
    final PathGenerator mPathGenerator;
    @NonNull
    private final Logger mLogger = new Logger("Callbacks");
    @NonNull
    private final IScreenChangedCallback mScreenChangedCallback;
    @NonNull
    Session mSession;
    @VisibleForTesting
    Activity mLiveActivity;
    @VisibleForTesting
    Handler mHandler;
    @VisibleForTesting
    Runnable mHideRunnable;
    @VisibleForTesting
    Runnable mExitRunnable;
    @VisibleForTesting
    long mExitTimestamp;

    /**
     * Constructs an Activity Callback for this session, and sends a Application Start event.
     * This callback is attached to the system by the caller.
     */
    public CsActivityCallbacks() {
        final SessionComponent sessionComponent = SingletonProvider.getSessionComponent();
        mScreenMonitoringService = sessionComponent.getScreenMonitoringService();
        mPathGenerator = sessionComponent.getPathGenerator();
        mSession = sessionComponent.getSession();
        sessionComponent.inject(this);
        mHandler = new Handler();
        mLogger.d("sending start event");
        mScreenChangedCallback = new ScreenChangedCallback(
                mPathGenerator,
                mSession,
                mLogger);
        AppStartEventBuilder start = mSession.getEventsFactory().produceEvent(APP_START);
        mSession.getEventsProcessor()
                .processEvent(mSession.getJsonProxy().serializeToJson(start.build()));
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        //NO-OP
    }

    @Override
    public void onActivityStarted(Activity activity) {
        //NO-OP
    }

    @DebugLog
    @Override
    public void onActivityResumed(Activity activity) {
        int loadTime;
        if (mExitTimestamp == 0) {
            // initial activity started. saying that load time is 0
            loadTime = 0;
            AppShowEventBuilder show = mSession.getEventsFactory().produceEvent(APP_SHOW);
            mSession.getEventsProcessor()
                    .processEvent(mSession.getJsonProxy().serializeToJson(show.build()));
        } else {
            loadTime = (int) (System.currentTimeMillis() - mExitTimestamp);
        }
        mLiveActivity = activity;

        String url = null;
        Window window = activity.getWindow();
        if (window != null) {
            FrameLayout rootView = window.findViewById(android.R.id.content);
            if (rootView != null) {
                url = mPathGenerator.generateUrl(activity, rootView);
            }
        }
        getSession().getGlass().setCurrentScreenUrl(url);

        getSession().attachGlassPane(mLiveActivity);

        mLogger.d("sending show event");

        if (mScreenMonitoringService.attachTo(mLiveActivity, mScreenChangedCallback)) {
            mLogger.d("ScreenMonitorService: the Activity %s is  being monitored.",
                    mLiveActivity.getTitle());
        }


        PageViewBuilder builder = getSession().produceEventBuilder(PAGE_VIEW);
        builder.setLoadTime(loadTime).setTitle(FrameworkUtility.getActivityTitle(activity));
        PageViewEvent event = builder.build();
        JsonProxy proxy = getSession().getJsonProxy();
        getSession().getEventsProcessor().processEvent(proxy.serializeToJson(event));

        if (mHideRunnable != null) {
            mLogger.d("canceling hide event event");
            mHandler.removeCallbacks(mHideRunnable);
        }

        if (mExitRunnable != null) {
            mLogger.d("canceling exit event event");
            mHandler.removeCallbacks(mExitRunnable);
        }
    }

    @DebugLog
    @Override
    public void onActivityPaused(Activity activity) {
        mExitTimestamp = System.currentTimeMillis();
        getSession().detachGlassPane(mLiveActivity);
        if (activity != null) {
            mScreenMonitoringService.detachFrom(activity);
        } else {
            mLogger.d("[onActivityPaused] : the Activity was null when trying to call "
                    + "#detachFrom on ScreenMonitoringService");
        }

        mLiveActivity = null;
        getSession().getGlass().setCurrentScreenUrl(null);

        mHideRunnable = new Runnable() {
            @Override
            public void run() {
                mLogger.d("sending hide event");
                AppHideEventBuilder hideBuilder = mSession.getEventsFactory()
                        .produceEvent(APP_HIDE);
                mSession.getEventsProcessor()
                        .processEvent(mSession.getJsonProxy().serializeToJson(hideBuilder.build()));
                mHideRunnable = null;
                getSession().getEventsProcessor().flushPendingEvents();
            }
        };
        mLogger.d("scheduling hide");
        mHandler.postDelayed(mHideRunnable, HIDE_TIMEOUT);

        mExitRunnable = new Runnable() {
            @Override
            public void run() {
                mLogger.d("sending exit event");
                AppExitEventBuilder exit = mSession.getEventsFactory().produceEvent(APP_EXIT);
                mSession.getEventsProcessor()
                        .processEvent(mSession.getJsonProxy().serializeToJson(exit.build()));
                mExitRunnable = null;
                getSession().getEventsProcessor().flushPendingEvents();
            }
        };
        mLogger.d("scheduling exit");
        mHandler.postDelayed(mExitRunnable, EXIT_TIMEOUT);

    }

    @Override
    public void onActivityStopped(Activity activity) {
        //NO-OP
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
        //NO-OP
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        //NO-OP
    }

    @NonNull
    public Session getSession() {
        return mSession;
    }

    public Activity getLiveActivity() {
        return mLiveActivity;
    }

    static class ScreenChangedCallback implements IScreenChangedCallback {

        @NonNull
        private final PathGenerator mPathGenerator;
        @NonNull
        private final Session mSession;
        @NonNull
        private final Logger mLogger;

        ScreenChangedCallback(@NonNull PathGenerator pathGenerator,
                              @NonNull Session session,
                              @NonNull Logger logger) {
            this.mPathGenerator = pathGenerator;
            this.mSession = session;
            this.mLogger = logger;
        }

        @Override
        public void onScreenChanged(@NonNull Activity activity,
                                    @NonNull Fragment fragment,
                                    int loadingTime) {

            final Window window = activity.getWindow();
            if (window != null) {
                final ViewGroup rootView = window.findViewById(android.R.id.content);

                // update the screen url
                String url = mPathGenerator.generateUrl(activity, fragment, rootView);
                mSession.getGlass().setCurrentScreenUrl(url);

                sendNewPageViewEvent(activity, mSession, loadingTime);

            } else {
                final String warningLogFormat = "[OnScreenChanged]: Was called for activity: [%s] "
                        + " but the activity did not have a Window";
                final String activityName = activity.getClass().getSimpleName();
                mLogger.w(warningLogFormat, activityName);
            }
        }

        private void sendNewPageViewEvent(@NonNull Activity activity,
                                          @NonNull Session session,
                                          int loadingTime) {
            PageViewBuilder builder = session.produceEventBuilder(PAGE_VIEW);
            builder.setLoadTime(loadingTime)
                    .setTitle(FrameworkUtility.getActivityTitle(activity));
            PageViewEvent event = builder.build();
            JsonProxy proxy = session.getJsonProxy();
            session.getEventsProcessor().processEvent(proxy.serializeToJson(event));
        }
    }
}
