package com.contentsquare.android.internal.logging;

import android.annotation.SuppressLint;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.BuildConfig;

/**
 * ContentSquare's Main logging Facility.
 * The preferred way of logging is to construct an instance of this class and use it locally in the
 * class you want to log events. Eventually the instance will be garbage collected.
 * One benefit in using the instance methods is that you have the string formater in place, which
 * means the mLogger will never concat strings. A very important aspect if we're to remain
 * lightweight in terms of memory consumption
 */
@SuppressLint("all")
public class Logger {
    @VisibleForTesting
    static boolean sLogEnabled = false;
    private final String mTag;

    /**
     * Constructor which will create an instance where all logs will be using the same tag.
     *
     * @param tag - the tag with which all logs will be logged..
     */
    public Logger(String tag) {
        mTag = tag;
    }

    /**
     * Checks if the log is enabled.
     *
     * @return - true if the log facility is enabled.
     */
    @VisibleForTesting
    public static boolean isLogEnabled() {
        return sLogEnabled;
    }

    /**
     * Enables or Disables the logging mechanism.
     *
     * @param enableLogs - property which defines the upcoming status of the logging system.
     */
    static void setLogEnabled(boolean enableLogs) {
        sLogEnabled = enableLogs;
    }

    /**
     * Initiates the Logging facility.
     */
    public static void init() {
        //if the library is built in debug mode, the default behaviour is ON.
        // CHECKSTYLE:OFF
        android.util.Log.e("cs", "LoggingFacility enabled : " + BuildConfig.DEBUG);
        // CHECKSTYLE:ON
        if (BuildConfig.DEBUG) {
            setLogEnabled(true);
        }
    }

    /**
     * Allows logs to be activated/deactivated manually.
     * Mainly used in god Mode.
     *
     * @param shouldLog boolean representing the new state.
     */
    public static void overrideLogging(boolean shouldLog) {
        setLogEnabled(shouldLog);
    }

// CHECKSTYLE:OFF

    /**
     * Send a {@link android.util.Log#VERBOSE} log message.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @return an int representing the status of the log
     */
    static int v(String tag, String msg) {
        if (sLogEnabled) {
            return android.util.Log.v(tag, msg);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#VERBOSE} log message and log the exception.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @param tr  An exception to log
     * @return an int representing the status of the log
     */
    static int v(String tag, String msg, Throwable tr) {
        if (sLogEnabled) {
            return android.util.Log.v(tag, msg, tr);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#DEBUG} log message.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @return an int representing the status of the log
     */
    static int d(String tag, String msg) {
        if (sLogEnabled) {
            return android.util.Log.d(tag, msg);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#DEBUG} log message and log the exception.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @param tr  An exception to log
     * @return an int representing the status of the log
     */
    static int d(String tag, String msg, Throwable tr) {
        if (sLogEnabled) {
            return android.util.Log.d(tag, msg, tr);
        }
        return -1;
    }

    /**
     * Send an {@link android.util.Log#INFO} log message.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @return an int representing the status of the log
     */
    static int i(String tag, String msg) {
        if (sLogEnabled) {
            return android.util.Log.i(tag, msg);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#INFO} log message and log the exception.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @param tr  An exception to log
     * @return an int representing the status of the log
     */
    static int i(String tag, String msg, Throwable tr) {
        if (sLogEnabled) {
            return android.util.Log.i(tag, msg, tr);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#WARN} log message.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @return an int representing the status of the log
     */
    static int w(String tag, String msg) {
        if (sLogEnabled) {
            return android.util.Log.w(tag, msg);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#WARN} log message and log the exception.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @param tr  An exception to log
     * @return an int representing the status of the log
     */
    static int w(String tag, String msg, Throwable tr) {
        if (sLogEnabled) {
            return android.util.Log.w(tag, msg, tr);
        }
        return -1;
    }


    // methods wrapping around android.util.Log's methods. Identical interface follows

    /**
     * Send a {@link android.util.Log#WARN} log message and log the exception.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param tr  An exception to log
     * @return an int representing the status of the log
     */
    static int w(String tag, Throwable tr) {
        if (sLogEnabled) {
            return android.util.Log.w(tag, "", tr);
        }
        return -1;
    }

    /**
     * Send an {@link android.util.Log#ERROR} log message.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @return an int representing the status of the log
     */
    static int e(String tag, String msg) {
        if (sLogEnabled) {
            return android.util.Log.e(tag, msg);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#ERROR} log message and log the exception.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param tag Used to identify the source of a log message.  It usually identifies
     *            the class or activity where the log call occurs.
     * @param msg The message you would like logged.
     * @param tr  An exception to log
     * @return an int representing the status of the log
     */
    static int e(String tag, String msg, Throwable tr) {
        if (sLogEnabled) {
            return android.util.Log.e(tag, msg, tr);
        }
        return -1;
    }

    /**
     * Send a {@link android.util.Log#DEBUG} log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void d(@NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            d(mTag, String.format(message, (Object[]) parameters));
        }
    }

    /**
     * Send a log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param t          - the throwable to be logged
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void d(Throwable t, @NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            d(mTag, String.format(message, (Object[]) parameters), t);
        }
    }

    /**
     * Send a {@link android.util.Log#INFO} log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void i(@NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            i(mTag, String.format(message, (Object[]) parameters));
        }
    }

    /**
     * Send a {@link android.util.Log#INFO} log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param t          - the throwable to be logged
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void i(Throwable t, @NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            i(mTag, String.format(message, (Object[]) parameters), t);
        }
    }

    /**
     * Send a {@link android.util.Log#WARN} log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void w(@NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            w(mTag, String.format(message, (Object[]) parameters));
        }
    }

    /**
     * Send a {@link android.util.Log#WARN} log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param t          - the throwable to be logged
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void w(Throwable t, @NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            w(mTag, String.format(message, (Object[]) parameters), t);
        }
    }

    /**
     * Send a {@link android.util.Log#ERROR} log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void e(@NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            e(mTag, String.format(message, (Object[]) parameters));
        }
    }

    /**
     * Send a {@link android.util.Log#ERROR} log message with the tag provided in the constructor.
     * The message is allowed to be in a formatted state using
     * {@link String#format(String, Object...)} the parameters which follow are the parameters
     * which will be forwarded to the format method.
     * Note:this method will not print any messages (or process the strings) unless in debug mode
     *
     * @param t          - the throwable to be logged
     * @param message    - The message you would like logged
     * @param parameters - a list of parameters for this string (to avoid concatenation)
     */
    public void e(Throwable t, @NonNull String message, Object... parameters) {
        if (sLogEnabled) {
            e(mTag, String.format(message, (Object[]) parameters), t);
        }
    }
// CHECKSTYLE:ON

}
