package com.contentsquare.android.internal.model;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.BuildConfig;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.model.data.ActionEvent;
import com.contentsquare.android.internal.model.data.AppExitEvent;
import com.contentsquare.android.internal.model.data.AppHideEvent;
import com.contentsquare.android.internal.model.data.AppShowEvent;
import com.contentsquare.android.internal.model.data.AppStartEvent;
import com.contentsquare.android.internal.model.data.CrashEvent;
import com.contentsquare.android.internal.model.data.DoubleTapEvent;
import com.contentsquare.android.internal.model.data.DragEvent;
import com.contentsquare.android.internal.model.data.EventsBundle;
import com.contentsquare.android.internal.model.data.FlickEvent;
import com.contentsquare.android.internal.model.data.LongPressEvent;
import com.contentsquare.android.internal.model.data.LowMemoryEvent;
import com.contentsquare.android.internal.model.data.PageViewEvent;
import com.contentsquare.android.internal.model.data.TapEvent;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Json Serialization proxy.
 */
public class JsonProxy {

    /**
     * Label used for the {@link java.util.UUID} descriptor of this event.
     */
    public static final String LABEL_EVENT_UUID = "euid";
    /**
     * Label used for the events {@link ActionEvent}.
     */
    public static final String LABEL_EVENT_ACTION = "ea";
    /**
     * Label used for this event's URL.
     */
    public static final String LABEL_URL = "url";
    /**
     * Label used for the Order number of this screen in this session.
     */
    static final String LABEL_SCREEN_NUMBER = "scn";
    /**
     * Label used for the connection type.
     */
    static final String LABEL_CONNECTION_TYPE = "c";
    /**
     * Label used for the carrier id.
     */
    static final String LABEL_CARRIER_ID = "ci";
    /**
     * Label used for the battery level.
     */
    static final String LABEL_BATTERY = "b";
    /**
     * Label used for the battery charging state.
     */
    static final String LABEL_BATTERY_CHARGING = "bc";
    /**
     * Label used for the orientation.
     */
    static final String LABEL_ORIENTATION = "o";
    /**
     * Label used for the version origin of this event.
     */
    static final String LABEL_VERSION_ORIGIN = "vo";
    /**
     * Label used for the session number.
     */
    static final String LABEL_SESSION_NUMBER = "sn";
    /**
     * Label used for the timestamp.
     */
    private static final String LABEL_TIMESTAMP = "t";
    /**
     * Label used for the screen title on a pageview.
     */
    static final String LABEL_PAGE_SCREEN_TITLE = "st";
    /**
     * Label used for the screen load time on a pageview.
     */
    static final String LABEL_PAGE_LOAD_TIME = "sl";
    /**
     * Label used for the message on a crash.
     */
    static final String LABEL_CRASH_MESSAGE = "m";
    /**
     * Label used for the crash origin .
     */
    static final String LABEL_CRASH_ORIGIN = "co";
    /**
     * Label used for the crash fatality .
     */
    static final String LABEL_CRASH_FATAL = "cf";

    /**
     * Label used for the Batch project id.
     */
    static final String BATCH_PROJECT_ID = "pid";

    /**
     * Label used for the Batch user id.
     */
    static final String BATCH_USER_ID = "uid";

    /**
     * Label used for the Batch device type.
     */
    static final String BATCH_DEVICE_TYPE = "dt";
    /**
     * Label used for the Batch OS name.
     */
    private static final String BATCH_OS_NAME = "os";
    /**
     * Label used for the Batch Language .
     */
    static final String BATCH_LANGUAGE = "l";
    /**
     * Label used for the Batch Anonimizer.
     */
    static final String BATCH_ANONIMIZER = "a";
    /**
     * Label used for the Batch Timezone.
     */
    static final String BATCH_TIMEZONE = "tz";
    /**
     * Label used for the Batch Type Origin.
     */
    private static final String BATCH_TYPE_ORIGIN = "to";
    /**
     * Label used for the Batch App name.
     */
    public static final String BATCH_APP_NAME = "an";
    /**
     * Label used for the Batch SDK build type.
     */
    public static final String BATCH_SDK_TYPE = "st";
    /**
     * Label used for the Batch SDK flavor .
     */
    public static final String BATCH_SDK_FLAVOR = "sf";
    /**
     * Label used for the Batch Resolution.
     */
    private static final String BATCH_DISPLAY_RESOLUTION = "r";
    /**
     * Label used for the Batch bundle paylooad.
     */
    public static final String BATCH_PAYLOAD = "pl";

    //TODO: Refactor label names (remove TAP)
    /**
     * Label used for the Tap event View path.
     */
    static final String LABEL_TAP_PATH = "tvp";
    /**
     * Label used for the Tap event view label.
     */
    static final String LABEL_TAP_LABEL = "tvt";
    /**
     * Label used for the Tap event for the view id.
     */
    static final String LABEL_TAP_VIEWID = "tvid";
    /**
     * Label used for the Tap event accessibility event.
     */
    static final String LABEL_TAP_ACS_LABEL = "tvac";

    /**
     * Label used for the finger direction [1,2,3,4,5] -> [up, down, left, right, complex_pattern]
     * for the different action events.
     */
    static final String LABEL_FINGER_DIRECTION = "fd";
    /**
     * Label used for the target view distance dragged for the different action events.
     */
    static final String LABEL_TARGET_VIEW_DISTANCE_DRAGGED = "tvd";
    /**
     * Label used for the target view velocity for the different action events.
     */
    static final String LABEL_TARGET_VIEW_VELOCITY = "tvv";
    /**
     * Label used for the Tap event responsiveness proeperty.
     */
    public static final String LABEL_TAP_UNRESPONSIVE = "ur";
    /**
     * Label used for the available memory parameter when a low memory event is dispatched.
     */
    public static final String LABEL_HEAP_MEMORY_LEVEL = "am";

    private final Logger mLog = new Logger("JsonProxy");


    /**
     * Provides a json representation of the common part of a data object.
     *
     * @return a {@link JSONObject} or null if there is a serialization issue.
     */
    @NonNull
    @VisibleForTesting
    JSONObject serializeSuper(@NonNull ActionEvent event) throws JSONException {

        JSONObject result = new JSONObject();
        result.put(LABEL_EVENT_UUID, event.getUuid());
        result.put(LABEL_EVENT_ACTION, event.getEventAction());
        result.put(LABEL_URL, event.getUrl());
        result.put(LABEL_SCREEN_NUMBER, event.getScreenCount());
        result.put(LABEL_CONNECTION_TYPE, event.getConnectionType());
        result.put(LABEL_CARRIER_ID, event.getCarrierId());
        result.put(LABEL_BATTERY, event.getBatteryLevel());
        result.put(LABEL_BATTERY_CHARGING, event.isBatteryCharging());
        result.put(LABEL_ORIENTATION, event.getOrientation());
        result.put(LABEL_VERSION_ORIGIN, event.getOriginVersion());
        result.put(LABEL_SESSION_NUMBER, event.getSessionNumber());
        result.put(LABEL_TIMESTAMP, event.getTimestamp());
        return result;
    }

    /**
     * Serializes {@link PageViewEvent} to {@link JSONObject}.
     *
     * @param event the PageView event
     * @return a JSon representation of the event, or a null in the case of a {@link JSONObject}.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull PageViewEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_PAGE_SCREEN_TITLE, event.getTitle());
            obj.put(LABEL_PAGE_LOAD_TIME, event.getLoadTime());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[PageViewEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes {@link CrashEvent} to {@link JSONObject}.
     *
     * @param event the Crash event
     * @return a JSon representation of the event, null in the case of an error.
     */
    @Nullable
    JSONObject serializeToJson(@NonNull CrashEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_CRASH_MESSAGE, event.getMessage());
            obj.put(LABEL_CRASH_ORIGIN, event.getCrashOrigin());
            obj.put(LABEL_CRASH_FATAL, event.isFatal());

            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[CrashEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes {@link TapEvent} to {@link JSONObject}.
     *
     * @param event the Tap event
     * @return a JSon representation of the event, null in the case of an error.
     */
    @Nullable
    JSONObject serializeToJson(@NonNull TapEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_TAP_PATH, event.getTouchPath());
            obj.put(LABEL_TAP_LABEL, event.getViewLabel());
            obj.put(LABEL_TAP_VIEWID, event.getViewId());
            obj.put(LABEL_TAP_ACS_LABEL, event.getViewAccessibilityLabel());
            obj.put(LABEL_TAP_UNRESPONSIVE, event.isTouchUnresponsive());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[TapEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes {@link DoubleTapEvent} to {@link JSONObject}.
     *
     * @param event the Double Tap event
     * @return a JSon representation of the event, null in the case of an error.
     */
    @Nullable
    JSONObject serializeToJson(@NonNull DoubleTapEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_TAP_PATH, event.getTouchPath());
            obj.put(LABEL_TAP_LABEL, event.getViewLabel());
            obj.put(LABEL_TAP_VIEWID, event.getViewId());
            obj.put(LABEL_TAP_ACS_LABEL, event.getViewAccessibilityLabel());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[DoubleTapEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes {@link LongPressEvent} to {@link JSONObject}.
     *
     * @param event the Long Press event
     * @return a Json presentation of the event, null in the case of an error.
     */
    @Nullable
    JSONObject serializeToJson(@NonNull LongPressEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_TAP_PATH, event.getTouchPath());
            obj.put(LABEL_TAP_LABEL, event.getViewLabel());
            obj.put(LABEL_TAP_VIEWID, event.getViewId());
            obj.put(LABEL_TAP_ACS_LABEL, event.getViewAccessibilityLabel());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[LongPressEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes {@link FlickEvent} to {@link JSONObject}.
     *
     * @param event the Flick event
     * @return a Json presentation of the event, null in the case of an error.
     */
    @Nullable
    JSONObject serializeToJson(@NonNull FlickEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_TAP_PATH, event.getTouchPath());
            obj.put(LABEL_TAP_LABEL, event.getViewLabel());
            obj.put(LABEL_TAP_VIEWID, event.getViewId());
            obj.put(LABEL_TAP_ACS_LABEL, event.getViewAccessibilityLabel());
            obj.put(LABEL_FINGER_DIRECTION, event.getFingerDirection());
            obj.put(LABEL_TARGET_VIEW_DISTANCE_DRAGGED, event.getTargetViewDistanceDragged());
            obj.put(LABEL_TARGET_VIEW_VELOCITY, event.getTargetViewVelocity());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[FlickEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes {@link DragEvent} to {@link JSONObject}.
     *
     * @param event the Drag event
     * @return a Json presentation of the event, null in the case of an error.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull DragEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_TAP_PATH, event.getTouchPath());
            obj.put(LABEL_TAP_LABEL, event.getViewLabel());
            obj.put(LABEL_TAP_VIEWID, event.getViewId());
            obj.put(LABEL_TAP_ACS_LABEL, event.getViewAccessibilityLabel());
            obj.put(LABEL_FINGER_DIRECTION, event.getFingerDirection());
            obj.put(LABEL_TARGET_VIEW_DISTANCE_DRAGGED, event.getTargetViewDistanceDragged());
            obj.put(LABEL_TARGET_VIEW_VELOCITY, event.getTargetViewVelocity());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[DragEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes the {@link LowMemoryEvent} to {@link JSONObject}
     *
     * @param event as the {@link LowMemoryEvent}
     * @return a Json representation of the event or null in case of parsing exception.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull LowMemoryEvent event) {
        try {
            JSONObject obj = serializeSuper(event);
            obj.put(LABEL_HEAP_MEMORY_LEVEL, event.getCurrentAvailableMemory());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[LowMemoryEvent] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }


    /**
     * Serializes {@link EventsBundle} to {@link JSONObject}.
     *
     * @param events the bundle of event
     * @return a JSon representation of the event.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull EventsBundle events) {
        try {
            JSONObject obj = new JSONObject();
            obj.put(BATCH_PROJECT_ID, events.getProjectId());
            obj.put(BATCH_USER_ID, events.getUserId());
            obj.put(BATCH_DEVICE_TYPE, events.getDeviceType());
            obj.put(BATCH_OS_NAME, events.getOsName());
            obj.put(BATCH_LANGUAGE, events.getDeviceLanguage());
            obj.put(BATCH_ANONIMIZER, events.isAnonymizerEnabled());
            obj.put(BATCH_TIMEZONE, events.getTimezone());
            obj.put(BATCH_TYPE_ORIGIN, events.getTypeOrigin());
            obj.put(BATCH_DISPLAY_RESOLUTION, events.getDeviceResolution());
            obj.put(BATCH_PAYLOAD, events.getPayload());
            return obj;
        } catch (JSONException e) {
            mLog.w(e, "[EventsBundle] Error in json proxy : %s", e.getMessage());
            return null;
        }
    }

    /**
     * Serializes {@link AppStartEvent} to {@link JSONObject}.
     *
     * @param event the bundle of event
     * @return a JSon representation of the event.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull AppStartEvent event) {
        JSONObject obj = null;
        try {
            obj = serializeSuper(event);
        } catch (JSONException e) {
            mLog.w(e, "[AppStartEvent] Error in json proxy : %s", e.getMessage());
        }
        return obj;
    }

    /**
     * Serializes {@link AppShowEvent} to {@link JSONObject}.
     *
     * @param event the bundle of event
     * @return a JSon representation of the event.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull AppShowEvent event) {
        JSONObject obj = null;
        try {
            obj = serializeSuper(event);
        } catch (JSONException e) {
            mLog.w(e, "[AppShowEvent] Error in json proxy : %s", e.getMessage());
        }
        return obj;
    }

    /**
     * Serializes {@link AppHideEvent} to {@link JSONObject}.
     *
     * @param event the bundle of event
     * @return a JSon representation of the event.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull AppHideEvent event) {
        JSONObject obj = null;
        try {
            obj = serializeSuper(event);
        } catch (JSONException e) {
            mLog.w(e, "[AppHideEvent] Error in json proxy : %s", e.getMessage());
        }
        return obj;
    }

    /**
     * Serializes {@link AppExitEvent} to {@link JSONObject}.
     *
     * @param event the bundle of event
     * @return a JSon representation of the event.
     */
    @Nullable
    public JSONObject serializeToJson(@NonNull AppExitEvent event) {
        JSONObject obj = null;
        try {
            obj = serializeSuper(event);
        } catch (JSONException e) {
            mLog.w(e, "[AppExitEvent] Error in json proxy : %s", e.getMessage());
        }
        return obj;
    }

    /**
     * Utility method to switch for abstract events.
     *
     * @param event the event
     * @return a json representation
     */
    @Nullable
    public JSONObject serializeToJson(@Nullable ActionEvent event) {

        if (event == null) {
            return null;
        }
        if (event instanceof PageViewEvent) {
            return serializeToJson((PageViewEvent) event);
        } else if (event instanceof TapEvent) {
            return serializeToJson((TapEvent) event);
        } else if (event instanceof DoubleTapEvent) {
            return serializeToJson((DoubleTapEvent) event);
        } else if (event instanceof LongPressEvent) {
            return serializeToJson((LongPressEvent) event);
        } else if (event instanceof DragEvent) {
            return serializeToJson((DragEvent) event);
        } else if (event instanceof FlickEvent) {
            return serializeToJson((FlickEvent) event);
        } else if (event instanceof CrashEvent) {
            return serializeToJson((CrashEvent) event);
        } else if (event instanceof AppStartEvent) {
            return serializeToJson((AppStartEvent) event);
        } else if (event instanceof AppShowEvent) {
            return serializeToJson((AppShowEvent) event);
        } else if (event instanceof AppHideEvent) {
            return serializeToJson((AppHideEvent) event);
        } else if (event instanceof AppExitEvent) {
            return serializeToJson((AppExitEvent) event);
        } else {
            if (BuildConfig.DEBUG) {
                mLog.e("!!Wrong event type sent! returning null.");
            }
            return null;
        }
    }
}
