package com.contentsquare.android.internal.model.data;

import android.support.annotation.IntDef;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.contentsquare.android.internal.DeviceInfo;
import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.util.Strings;

import org.json.JSONObject;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.UUID;

/**
 * An abstract event describing the common header of All Action events, described in the confluence
 * specs.
 * All events have a common part which is an event header, which contains general info on the state
 * of the device at the moment the event happened.
 *
 * @see <a href="https://contentsquare.atlassian.net/wiki/spaces/RD/pages/54919203/Events+Format#EventsFormat-EventBatching">
 * Confluence specs</a>
 */
public abstract class ActionEvent {


    /**
     * Event Action for Error cases.
     */
    public static final int NO_EVENT = -1;

    /**
     * Event Action for Application Start events.
     */
    public static final int APP_START = 0;

    /**
     * Event Action for Application Show events.
     */
    public static final int APP_SHOW = 1;

    /**
     * Event Action for Application Hide events.
     */
    public static final int APP_HIDE = 2;

    /**
     * Event Action for Application exit events.
     */
    public static final int APP_EXIT = 3;

    /**
     * Event Action for Pageview events.
     */
    public static final int PAGE_VIEW = 4;

    /**
     * Event Action for Screen rotate or resize events.
     */
    public static final int SCREEN_ROTATE_RESIZE = 5;

    /**
     * Event Action for Tap/click events.
     * Touch Down → N x Touch Move → Touch Up
     * Duration : < 200ms
     * Distance : < 24 px
     */
    public static final int TAP = 6;

    /**
     * Event Action for Double Tap/click events.
     * Touch Down → N x Touch Move → Touch Up → Touch Down → N x Touch Move → Touch Up
     * Duration : < 200ms
     * Distance : < 24 px
     * Time between two taps < 200ms
     */
    public static final int DOUBLE_TAP = 7;

    /**
     * Event Action for Longpress events.
     * Touch Down → N x Touch Move → Touch Up
     * Duration : > 200ms
     * Distance : < 24 px
     */
    public static final int LONG_PRESS = 8;

    /**
     * Event Action for Drag events.
     * Touch Down → N x Touch Move → Touch Up
     * Duration : > 200ms
     * Distance : > 48 px
     * Finger Velocity < 100 px/s
     */
    public static final int DRAG = 9;

    /**
     * Event Action for flick events.
     * Touch Down → N x Touch Move → Touch Up
     * Duration : > 200ms
     * Distance : > 48 px
     * Finger Velocity > 100 px/s
     */
    public static final int FLICK = 10;

    /**
     * Event Action for Pinch events.
     * Multi finger Touch Down → N x Touch Move → Touch Up
     * Duration : > 200ms
     * Distance : > 48 px
     * Finger Velocity < 100 px/s
     */
    public static final int PINCH = 11;

    /**
     * Event Action for Keyboard show/hide events.
     */
    public static final int KEYBOARD_EVENT = 12;

    /**
     * Event Action for Keyboard input events.
     */
    public static final int INPUT = 13;

    /**
     * Event Action for Install referral events.
     */
    public static final int INSTALL_REFERRAL = 14;

    /**
     * Event Action for Custom client events.
     */
    public static final int CUSTOM_EVENT = 15;

    /**
     * Event Action for Financial transaction events.
     */
    public static final int TRANSACTION = 16;

    /**
     * Event Action for Crash events.
     */
    public static final int CRASH = 17;

    /**
     * Event Action for Low memory events.
     */
    public static final int LOW_MEMORY = 18;

    /**
     * Orientation type : portrait.
     */
    public static final int ORIENTATION_PORTRAIT = 0;

    /**
     * Orientation type : landscape.
     */
    public static final int ORIENTATION_LANDSCAPE = 1;

    /**
     * Finger direction : null.
     */
    public static final int FINGER_DIRECTION_NULL = 0;

    /**
     * Finger direction : up.
     */
    public static final int FINGER_DIRECTION_UP = 1;

    /**
     * Finger direction : down.
     */
    public static final int FINGER_DIRECTION_DOWN = 2;

    /**
     * Finger direction : left.
     */
    public static final int FINGER_DIRECTION_LEFT = 3;

    /**
     * Finger direction : right.
     */
    public static final int FINGER_DIRECTION_RIGHT = 4;

    /**
     * Finger direction : complex pattern.
     */
    public static final int FINGER_DIRECTION_COMPLEX_PATTERN = 5;

    /**
     * Connectivity type Offline.
     */
    public static final int CONNECTIVITY_ERROR = 0;

    /**
     * Connectivity type Offline.
     */
    public static final int OFFLINE = 1;
    /**
     * Connectivity type Wifi.
     */
    public static final int WIFI = 2;
    /**
     * Connectivity type edge.
     */
    public static final int EDGE = 3;
    /**
     * Connectivity type HSPA/HSPA+.
     */
    public static final int HSPA = 4;
    /**
     * Connectivity type LTE.
     */
    public static final int LTE = 5;
    /**
     * Battery full state.
     */
    private static final int BATTERY_FULL = 100;
    /**
     * This Event's {@link UUID} identifier.
     */
    private final String mUuid;
    /**
     * The action of this event.
     */
    @EventAction
    private final int mEventAction;

    /**
     * page url: app-and://identifier/path?query=param".
     */
    @NonNull
    private final String mUrl;

    /**
     * Screen number since session start - autoincrement.
     */
    @IntRange(from = 0, to = Integer.MAX_VALUE)
    private final int mScreenCount;

    /**
     * Connection type descriptor(wifi/3g/lte).
     */
    @ConnectionType
    private final int mConnectionType;

    /**
     * Carrier_id if 3g, and ssid if wifi.
     */
    @NonNull
    private final String mCarrierId;

    /**
     * device battery level (int between -1(special val, can't retrieve); 0 - 100 ).
     */
    @IntRange(from = 0, to = BATTERY_FULL)
    private final int mBatteryLevel;

    /**
     * battery is charging.
     */
    private final boolean mBatteryCharging;

    /**
     * Device orientation.
     */
    @OrientationType
    private final int mOrientation;

    /**
     * Json Object describing the origin of the event.
     * <pre><code>
     * vo:{ // object - version origin of the event
     *       sv:"1.1.0", // string version of the sdk
     *       sb:4, // int - sdk build number
     *       av:"appversion", // String - application version
     *       af:"appflavor" // String - application string - [release/debug/god]
     * }
     * </code></pre>
     */
    @NonNull
    private final JSONObject mOriginVersion;

    /**
     * Session id (positive int).
     */
    @IntRange(from = 0, to = Integer.MAX_VALUE)
    private final int mSessionNumber;
    /**
     * Timestamp of the event.
     */
    private final long mTimestamp;

    /**
     * Constructs an instance of this object, provided the builder.
     *
     * @param builder the ActionEventBuilder.
     */
    ActionEvent(Builder builder) {
        mUuid = UUID.randomUUID().toString();
        mEventAction = builder.getEventAction();
        mUrl = builder.getUrl();
        mScreenCount = builder.getScreenCount();
        mConnectionType = builder.getConnectionType();
        mCarrierId = builder.getCarrierId();
        mBatteryLevel = builder.getBatteryLevel();
        mBatteryCharging = builder.isBatteryCharging();
        mOrientation = builder.getOrientation();
        mOriginVersion = builder.getOriginVersion();
        mSessionNumber = builder.getSessionNumber();
        mTimestamp = builder.getTimestamp();
    }

    public String getUuid() {
        return mUuid;
    }

    @EventAction
    public int getEventAction() {
        return mEventAction;
    }

    @NonNull
    public String getUrl() {
        return mUrl;
    }

    public int getScreenCount() {
        return mScreenCount;
    }

    @ConnectionType
    public int getConnectionType() {
        return mConnectionType;
    }

    @NonNull
    public String getCarrierId() {
        return mCarrierId;
    }

    public int getBatteryLevel() {
        return mBatteryLevel;
    }

    public boolean isBatteryCharging() {
        return mBatteryCharging;
    }

    @OrientationType
    public int getOrientation() {
        return mOrientation;
    }

    @NonNull
    public JSONObject getOriginVersion() {
        return mOriginVersion;
    }

    public int getSessionNumber() {
        return mSessionNumber;
    }

    public long getTimestamp() {
        return mTimestamp;
    }

    /**
     * Annotation limiting an int to only event actions scope.
     */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef( {
            NO_EVENT,
            APP_START,
            APP_SHOW,
            APP_HIDE,
            APP_EXIT,
            PAGE_VIEW,/*,
            SCREEN_ROTATE_RESIZE,*/
            TAP,
            DOUBLE_TAP,
            LONG_PRESS,
            CRASH,
            DRAG,
            FLICK,
            /* // disabled as out of scope for MVP
            PINCH,
            KEYBOARD_EVENT,
            INPUT,
            INSTALL_REFERRAL,
            CUSTOM_EVENT,
            TRANSACTION,
            LOW_MEMORY*/})
    public @interface EventAction {
    }


    /**
     * Annotation limiting an int to only Finger related actions.
     */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef( {
            NO_EVENT,
            TAP,
            DOUBLE_TAP,
            LONG_PRESS,
            DRAG,
            FLICK,
            PINCH})
    public @interface FingerAction {
    }

    /**
     * Annotation limiting an int to only Finger related actions.
     */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef({
            FINGER_DIRECTION_NULL,
            FINGER_DIRECTION_UP,
            FINGER_DIRECTION_DOWN,
            FINGER_DIRECTION_LEFT,
            FINGER_DIRECTION_RIGHT,
            FINGER_DIRECTION_COMPLEX_PATTERN})
    public @interface FingerDirection {
    }

    /**
     * Connection type int annotation.
     */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef( {CONNECTIVITY_ERROR, OFFLINE, WIFI, EDGE, HSPA, LTE})
    public @interface ConnectionType {
    }


    /**
     * Annotation limiting an int to two orientation values.
     */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef( {ORIENTATION_PORTRAIT, ORIENTATION_LANDSCAPE})
    public @interface OrientationType {
    }

    /**
     * A builder for Events.
     */
    public abstract static class Builder<T extends ActionEvent> {
        @EventAction
        private int mEventAction = CRASH;
        @NonNull
        private String mUrl = "";
        @IntRange(from = 0, to = Integer.MAX_VALUE)
        private int mScreenCount = 0;
        @ConnectionType
        private int mConnectionType = CONNECTIVITY_ERROR;
        @NonNull
        private String mCarrierId = "";
        @IntRange(from = 0, to = BATTERY_FULL)
        private int mBatteryLevel;
        private boolean mBatteryCharging;
        @OrientationType
        private int mOrientation;
        @NonNull
        private JSONObject mOriginVersion = new JSONObject();
        @IntRange(from = 0, to = Integer.MAX_VALUE)
        private int mSessionNumber;
        private long mTimestamp;

        /**
         * Constructs a builder for this type.
         */
        Builder() {
            //TODO : here, extract all the variables and states required from the Singletons.
            DeviceInfo deviceInfo = SingletonProvider.getAppComponent().getDeviceInfo();
            setBatteryLevel(deviceInfo.getBatteryLevelNow());
            setBatteryCharging(deviceInfo.isDeviceCharging());
            setCarrierId(deviceInfo.getCarrierName());
            setConnectionType(deviceInfo.getActiveConnectionType());
            setOrientation(deviceInfo.getScreenOrientation());
            setOriginVersion(deviceInfo.getVersionOrigin());

            Session session = SingletonProvider.getSessionComponent().getSession();
            setScreenCount(session.getScreenCount());
            setSessionNumber(session.getSessionId());
            setTimestamp(System.currentTimeMillis());
            final String currentScreenUrl = session.getGlass().getCurrentScreenUrl();
            if (!Strings.isNullOrEmpty(currentScreenUrl)) {
                setUrl(currentScreenUrl);
            }
        }

        //CHECKSTYLE:OFF
        @EventAction
        public int getEventAction() {
            return mEventAction;
        }

        public void setEventAction(int eventAction) {
            mEventAction = eventAction;
        }

        @Nullable
        public String getUrl() {
            return mUrl;
        }

        @NonNull
        public Builder setUrl(@NonNull String url) {
            this.mUrl = url;
            return this;
        }

        public int getScreenCount() {
            return mScreenCount;
        }

        @NonNull
        public Builder setScreenCount(int screenCount) {
            mScreenCount = screenCount;
            return this;
        }

        @ConnectionType
        public int getConnectionType() {
            return mConnectionType;
        }

        // TODO: 10/24/17 This Builder should be refactored and use generics for the return type on
        // these functions. Right now these functions are returning the abstract Builder and calling
        // build() on abstract Builder does not return the wanted Event type.
        @NonNull
        public Builder setConnectionType(@ConnectionType int connectionType) {
            mConnectionType = connectionType;
            return this;
        }

        @NonNull
        public String getCarrierId() {
            return mCarrierId;
        }

        @NonNull
        public Builder setCarrierId(@NonNull String carrierId) {
            mCarrierId = carrierId;
            return this;
        }

        @IntRange(from = 0, to = BATTERY_FULL)
        public int getBatteryLevel() {
            return mBatteryLevel;
        }

        @NonNull
        public Builder setBatteryLevel(@IntRange(from = 0, to = BATTERY_FULL) int batteryLevel) {
            mBatteryLevel = batteryLevel;
            return this;
        }

        public boolean isBatteryCharging() {
            return mBatteryCharging;
        }

        @NonNull
        public Builder setBatteryCharging(boolean batteryCharging) {
            mBatteryCharging = batteryCharging;
            return this;
        }

        @OrientationType
        public int getOrientation() {
            return mOrientation;
        }

        @NonNull
        public Builder setOrientation(@OrientationType int orientation) {
            mOrientation = orientation;
            return this;
        }

        @NonNull
        public JSONObject getOriginVersion() {
            return mOriginVersion;
        }

        @NonNull
        public Builder setOriginVersion(@NonNull JSONObject originVersion) {
            mOriginVersion = originVersion;
            return this;
        }

        public int getSessionNumber() {
            return mSessionNumber;
        }

        @NonNull
        public Builder setSessionNumber(
                @IntRange(from = 0, to = Integer.MAX_VALUE) int sessionNumber) {
            mSessionNumber = sessionNumber;
            return this;
        }

        public long getTimestamp() {
            return mTimestamp;
        }

        @NonNull
        public Builder setTimestamp(long timestamp) {
            mTimestamp = timestamp;
            return this;
        }
        //CHECKSTYLE:ON

        /**
         * Builds an actionEvent.
         *
         * @return the event.
         */
        @NonNull
        public abstract T build();

    }
}
