package com.contentsquare.android.internal.model.data;

import android.support.annotation.IntDef;
import android.support.annotation.NonNull;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * This is an event which describes a handled or unhandled crash.
 */
public final class CrashEvent extends ActionEvent {


    /**
     * Enum const depicting a crash happened, but we don't know the origin.
     */
    public static final int CRASH_UNKNOWN = 0;

    /**
     * Enum const depicting a crash happened coming from the sdk.
     */
    public static final int CRASH_SDK = 1;

    /**
     * Enum const depicting a crash happened, coming from the app.
     */
    public static final int CRASH_APP = 2;

    private final String mMessage;
    @CrashOrigin
    private final int mCrashOrigin;

    /**
     * Depicting if the report was fatal or was a developer log.
     */
    private final boolean mFatal;

    /**
     * Constructs an Event of the appropriate type.
     */
    private CrashEvent(CrashEventBuilder builder) {
        super(builder);
        mMessage = builder.getMessage();
        mCrashOrigin = builder.getCrashOrigin();
        mFatal = builder.isFatal();
    }

    public String getMessage() {
        return mMessage;
    }

    public int getCrashOrigin() {
        return mCrashOrigin;
    }

    public boolean isFatal() {
        return mFatal;
    }

    /**
     * Device type int annotation.
     */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef({CRASH_UNKNOWN, CRASH_SDK, CRASH_APP})
    public @interface CrashOrigin {
    }

    /**
     * Builder class for a Crash Event.
     */
    public static class CrashEventBuilder extends Builder<CrashEvent> {

        private String mMessage;
        @CrashOrigin
        private int mCrashOrigin;
        private boolean mFatal;

        /**
         * Constructs an Event of the type crash.
         */
        public CrashEventBuilder() {
            setEventAction(CRASH);
        }

        public boolean isFatal() {
            return mFatal;
        }

        public void setFatal(boolean fatal) {
            mFatal = fatal;
        }

        public String getMessage() {
            return mMessage;
        }

        //CHECKSTYLE:OFF
        public CrashEventBuilder setMessage(String message) {
            mMessage = message;
            return this;
        }
        //CHECKSTYLE:ON

        @CrashOrigin
        public int getCrashOrigin() {
            return mCrashOrigin;
        }

        //CHECKSTYLE:OFF
        public CrashEventBuilder setCrashOrigin(@CrashOrigin int crashOrigin) {
            mCrashOrigin = crashOrigin;
            return this;
        }
        //CHECKSTYLE:ON

        @Override
        @NonNull
        public CrashEvent build() {
            return new CrashEvent(this);

        }

    }
}