package com.contentsquare.android.internal.model.data;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

/**
 * This is an event which describes the Drag event.
 * This event is sent when a user interacts with the screen, performing a drag.
 */
public final class DragEvent extends ActionEvent {

    @NonNull
    private final String mTouchPath;
    @Nullable
    private final String mViewLabel;
    @Nullable
    private final String mViewId;
    @Nullable
    private final String mViewAccessibilityLabel;
    private final int mFingerDirection;
    private final int mTargetViewDistanceDragged;
    private final int mTargetViewVelocity;

    /**
     * Constructs an Event of Drag type, provided the builder.
     *
     * @param builder the ActionEventBuilder for DragEvent.
     */
    public DragEvent(DragEventBuilder builder) {
        super(builder);
        mTouchPath = builder.getTouchPath();
        mViewId = builder.getViewId();
        mViewLabel = builder.getViewLabel();
        mViewAccessibilityLabel = builder.getViewAccessibilityLabel();
        mFingerDirection = builder.getFingerDirection();
        mTargetViewDistanceDragged = builder.getTargetViewDistanceDragged();
        mTargetViewVelocity = builder.getTargetViewVelocity();
    }

    @NonNull
    public String getTouchPath() {
        return mTouchPath;
    }

    @Nullable
    public String getViewLabel() {
        return mViewLabel;
    }

    @Nullable
    public String getViewId() {
        return mViewId;
    }

    @Nullable
    public String getViewAccessibilityLabel() {
        return mViewAccessibilityLabel;
    }

    public int getFingerDirection() {
        return mFingerDirection;
    }

    public int getTargetViewDistanceDragged() {
        return mTargetViewDistanceDragged;
    }

    public int getTargetViewVelocity() {
        return mTargetViewVelocity;
    }

    /**
     * Builder class for a Drag Event.
     */
    public static class DragEventBuilder extends Builder<DragEvent> {

        @NonNull
        private String mTouchPath;
        @Nullable
        private String mViewLabel;
        @Nullable
        private String mViewId;
        @Nullable
        private String mViewAccessibilityLabel;
        private int mFingerDirection;
        private int mTargetViewDistanceDragged;
        private int mTargetViewVelocity;

        /**
         * Constructs an Event of the type Drag.
         */
        public DragEventBuilder() {
            super();
            setEventAction(DRAG);
        }

        //CHECKSTYLE:OFF
        @NonNull
        public String getTouchPath() {
            return mTouchPath;
        }

        @NonNull
        public DragEventBuilder setTouchPath(@NonNull String touchPath) {
            mTouchPath = touchPath;
            return this;
        }

        @Nullable
        public String getViewLabel() {
            return mViewLabel;
        }

        @NonNull
        public DragEventBuilder setViewLabel(@Nullable String viewLabel) {
            mViewLabel = viewLabel;
            return this;
        }

        @Nullable
        public String getViewId() {
            return mViewId;
        }

        @NonNull
        public DragEventBuilder setViewId(@Nullable String viewId) {
            mViewId = viewId;
            return this;
        }

        @Nullable
        public String getViewAccessibilityLabel() {
            return mViewAccessibilityLabel;
        }

        @NonNull
        public DragEventBuilder setViewAccessibilityLabel(
                @Nullable String viewAccessibilityLabel) {
            mViewAccessibilityLabel = viewAccessibilityLabel;
            return this;
        }

        public int getFingerDirection() {
            return mFingerDirection;
        }

        @NonNull
        public DragEventBuilder setFingerDirection(int fingerDirection) {
            mFingerDirection = fingerDirection;
            return this;
        }

        public int getTargetViewDistanceDragged() {
            return mTargetViewDistanceDragged;
        }

        @NonNull
        public DragEventBuilder setTargetViewDistanceDragged(int targetViewDistanceDragged) {
            mTargetViewDistanceDragged = targetViewDistanceDragged;
            return this;
        }

        public int getTargetViewVelocity() {
            return mTargetViewVelocity;
        }

        @NonNull
        public DragEventBuilder setTargetViewVelocity(int targetViewVelocity) {
            mTargetViewVelocity = targetViewVelocity;
            return this;
        }

        //CHECKSTYLE:ON

        @NonNull
        @Override
        public DragEvent build() {
            return new DragEvent(this);
        }
    }
}
