package com.contentsquare.android.internal.model.data;

import android.support.annotation.NonNull;

import com.contentsquare.android.internal.DeviceInfo;
import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.dagger.SingletonProvider;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.List;

/**
 * A bundle of events, combined in a {@link org.json.JSONObject}. As defined in the specs page.
 * A bundle is defined by a common header and a payload, comprised of multiple {@link ActionEvent}s.
 *
 * @see
 * <a href="https://contentsquare.atlassian.net/wiki/spaces/RD/pages/54919203/Events+Format#EventsFormat-EventBatching">
 * Confluence specs</a>
 */
public class EventsBundle {

    private final String mUserId;
    private final int mProjectId;
    private final int mDeviceType;
    private final String mOsName;
    private final String mDeviceLanguage;
    private final boolean mAnonimizerEnabled;
    private final String mTimezone;
    private final JSONObject mTypeOrigin;
    private final JSONObject mDeviceResolution;
    private final JSONArray mPayload;

    /**
     * Builds a Bundle using the provided builder.
     *
     * @param bundleBuilder the builder.
     */
    private EventsBundle(@NonNull BundleBuilder bundleBuilder) {
        mUserId = bundleBuilder.getUserId();
        mProjectId = bundleBuilder.getProjectId();
        mDeviceType = bundleBuilder.getDeviceType();
        mOsName = bundleBuilder.getOsName();
        mDeviceLanguage = bundleBuilder.getDeviceLanguage();
        mAnonimizerEnabled = bundleBuilder.isAnonymizerEnabled();
        mTimezone = bundleBuilder.getTimezone();
        mTypeOrigin = bundleBuilder.getTypeOrigin();
        mDeviceResolution = bundleBuilder.getDeviceResolution();
        mPayload = bundleBuilder.getPayload();

    }

    public String getUserId() {
        return mUserId;
    }

    public int getProjectId() {
        return mProjectId;
    }

    public int getDeviceType() {
        return mDeviceType;
    }

    public String getOsName() {
        return mOsName;
    }

    public String getDeviceLanguage() {
        return mDeviceLanguage;
    }

    public boolean isAnonymizerEnabled() {
        return mAnonimizerEnabled;
    }

    public String getTimezone() {
        return mTimezone;
    }

    public JSONObject getTypeOrigin() {
        return mTypeOrigin;
    }

    public JSONObject getDeviceResolution() {
        return mDeviceResolution;
    }

    public JSONArray getPayload() {
        return mPayload;
    }

    /**
     * A builder for NetworkBundle events.
     */
    public static final class BundleBuilder {
        private final String mUserId;
        private final int mProjectId;
        private final int mDeviceType;
        private final String mOsName;
        private final String mDeviceLanguage;
        private final boolean mAnonymizerEnabled;
        private final String mTimezone;
        private final JSONObject mTypeOrigin;
        private final JSONObject mDeviceResolution;
        private final JSONArray mPayload;

        /**
         * Builds a builder for bundles.
         */
        public BundleBuilder() {
            Session session = SingletonProvider.getSessionComponent().getSession();
            mUserId = session.getUserId();
            mProjectId = session.getRunConfiguration().getCsProjectId();
            mAnonymizerEnabled = session.getRunConfiguration().isAnonymousEnabled();
            DeviceInfo deviceInfo = SingletonProvider.getAppComponent().getDeviceInfo();
            mDeviceType = deviceInfo.getDeviceIntType();
            mOsName = deviceInfo.getDeviceOs();
            mDeviceLanguage = deviceInfo.getUserLanguage();
            mTimezone = deviceInfo.getUserTimezone();
            mTypeOrigin = deviceInfo.getTypeOrigin();
            mDeviceResolution = deviceInfo.getDeviceResolutionJson();
            mPayload = new JSONArray();
        }

        public String getUserId() {
            return mUserId;
        }

        public int getProjectId() {
            return mProjectId;
        }

        public int getDeviceType() {
            return mDeviceType;
        }

        public String getOsName() {
            return mOsName;
        }

        public String getDeviceLanguage() {
            return mDeviceLanguage;
        }

        public boolean isAnonymizerEnabled() {
            return mAnonymizerEnabled;
        }

        public String getTimezone() {
            return mTimezone;
        }

        public JSONObject getTypeOrigin() {
            return mTypeOrigin;
        }

        public JSONObject getDeviceResolution() {
            return mDeviceResolution;
        }

        public JSONArray getPayload() {
            return mPayload;
        }

        /**
         * Set a bundle payload.
         *
         * @param payload A list of JSon objects as payload.
         * @return this builder.
         */
        @NonNull
        public BundleBuilder setPayload(List<JSONObject> payload) {
            for (JSONObject json : payload) {
                mPayload.put(json);
            }
            return this;
        }

        /**
         * Builds an EventBundle with the builder.
         *
         * @return the EventBundle.
         */
        public EventsBundle build() {
            return new EventsBundle(this);
        }
    }

}
