package com.contentsquare.android.internal.model.data;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

/**
 * This is an event which describes the Flick event.
 * This event is sent when a user interacts with the screen, performing a flick.
 */
public final class FlickEvent extends ActionEvent {

    @NonNull
    private final String mTouchPath;
    @Nullable
    private final String mViewLabel;
    @Nullable
    private final String mViewId;
    @Nullable
    private final String mViewAccessibilityLabel;
    private final int mFingerDirection;
    private final int mTargetViewDistanceDragged;
    private final int mTargetViewVelocity;

    /**
     * Constructs an Event of the Flick type.
     *
     * @param builder the ActionEventBuilder for FlickEvent.
     */
    private FlickEvent(FlickEventBuilder builder) {
        super(builder);
        mTouchPath = builder.getTouchPath();
        mViewId = builder.getViewId();
        mViewLabel = builder.getViewLabel();
        mViewAccessibilityLabel = builder.getViewAccessibilityLabel();
        mFingerDirection = builder.getFingerDirection();
        mTargetViewDistanceDragged = builder.getTargetViewDistanceDragged();
        mTargetViewVelocity = builder.getTargetViewVelocity();
    }

    @NonNull
    public String getTouchPath() {
        return mTouchPath;
    }

    @Nullable
    public String getViewLabel() {
        return mViewLabel;
    }

    @Nullable
    public String getViewId() {
        return mViewId;
    }

    @Nullable
    public String getViewAccessibilityLabel() {
        return mViewAccessibilityLabel;
    }

    public int getFingerDirection() {
        return mFingerDirection;
    }

    public int getTargetViewDistanceDragged() {
        return mTargetViewDistanceDragged;
    }

    public int getTargetViewVelocity() {
        return mTargetViewVelocity;
    }

    /**
     * Builder class for a Flick Event.
     */
    public static class FlickEventBuilder extends Builder<FlickEvent> {

        @NonNull
        private String mTouchPath;
        @Nullable
        private String mViewLabel;
        @Nullable
        private String mViewId;
        @Nullable
        private String mViewAccessibilItyLabel;
        private int mFingerDirection;
        private int mTargetViewDistanceDragged;
        private int mTargetViewVelocity;

        /**
         * Constructs an Event of the type Flick.
         */
        public FlickEventBuilder() {
            super();
            setEventAction(FLICK);
        }

        //CHECKSTYLE:OFF
        @NonNull
        public String getTouchPath() {
            return mTouchPath;
        }

        @NonNull
        public FlickEventBuilder setTouchPath(@NonNull String touchPath) {
            mTouchPath = touchPath;
            return this;
        }

        @Nullable
        public String getViewLabel() {
            return mViewLabel;
        }

        @NonNull
        public FlickEventBuilder setViewLabel(@Nullable String viewLabel) {
            mViewLabel = viewLabel;
            return this;
        }

        @Nullable
        public String getViewId() {
            return mViewId;
        }

        @NonNull
        public FlickEventBuilder setViewId(@Nullable String viewId) {
            mViewId = viewId;
            return this;
        }

        @Nullable
        public String getViewAccessibilityLabel() {
            return mViewAccessibilItyLabel;
        }

        @NonNull
        public FlickEventBuilder setViewAccessibilityLabel(@Nullable String viewAccessibiltyLabel) {
            mViewAccessibilItyLabel = viewAccessibiltyLabel;
            return this;
        }

        public int getFingerDirection() {
            return mFingerDirection;
        }

        public FlickEventBuilder setFingerDirection(int fingerDirection) {
            mFingerDirection = fingerDirection;
            return this;
        }

        public int getTargetViewDistanceDragged() {
            return mTargetViewDistanceDragged;
        }

        public FlickEventBuilder setTargetViewDistanceDragged(int targetViewDistanceDragged) {
            mTargetViewDistanceDragged = targetViewDistanceDragged;
            return this;
        }

        public int getTargetViewVelocity() {
            return mTargetViewVelocity;
        }

        public FlickEventBuilder setTargetViewVelocity(int targetViewVelocity) {
            mTargetViewVelocity = targetViewVelocity;
            return this;
        }

        //CHECKSTYLE:ON

        @Override
        @NonNull
        public FlickEvent build() {
            return new FlickEvent(this);

        }

    }
}