package com.contentsquare.android.internal.model.data;

import android.support.annotation.NonNull;

/**
 * Event which notifies the interested party (server logs) that the current android process
 * is running out of memory.
 */

public class LowMemoryEvent extends ActionEvent {

    /**
     * The current available free HEAP memory in the application process.
     */
    private int mCurrentAvailableMemory;

    private LowMemoryEvent(@NonNull LowMemoryEventBuilder builder) {
        super(builder);
        mCurrentAvailableMemory = builder.mCurrentAvailableMemory;
    }

    /**
     * Provides the process free heap memory level in KB.
     *
     * @return the memory level in KB.
     */
    public int getCurrentAvailableMemory() {
        return mCurrentAvailableMemory;
    }

    /**
     * Builder class for the {@link LowMemoryEvent}.
     */
    public static class LowMemoryEventBuilder extends Builder<LowMemoryEvent> {

        private int mCurrentAvailableMemory = 0;

        /**
         * Generates a new instance of this builder class.
         */
        public LowMemoryEventBuilder() {
            super();
            setEventAction(LOW_MEMORY);
        }

        //CHECKSTYLE:OFF
        public LowMemoryEventBuilder currentAvailableMemory(int memoryLevelInKb) {
            mCurrentAvailableMemory = memoryLevelInKb;
            return this;
        }
        //CHECKSTYLE:ON

        @NonNull
        @Override
        public LowMemoryEvent build() {
            return new LowMemoryEvent(this);
        }
    }
}
