package com.contentsquare.android.internal.model.data;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

/**
 * This is an event which describes the Tap event.
 * This event is sent when a user interacts with the screen, performing a tap.
 */
public final class TapEvent extends ActionEvent {

    @NonNull
    private final String mTouchPath;
    @Nullable
    private final String mViewLabel;
    @Nullable
    private final String mViewId;
    @Nullable
    private final String mViewAccessibilityLabel;
    @Nullable
    private final boolean mTouchUnresponsive;


    /**
     * Constructs an Event of the Tap type.
     */
    private TapEvent(TapEventBuilder builder) {
        super(builder);
        mTouchPath = builder.getTouchPath();
        mViewId = builder.getViewId();
        mViewLabel = builder.getViewLabel();
        mViewAccessibilityLabel = builder.getViewAccessibiltyLabel();
        mTouchUnresponsive = builder.isTouchUnresponsive();
    }

    @NonNull
    public String getTouchPath() {
        return mTouchPath;
    }

    @Nullable
    public String getViewLabel() {
        return mViewLabel;
    }

    @Nullable
    public String getViewId() {
        return mViewId;
    }

    @Nullable
    public String getViewAccessibilityLabel() {
        return mViewAccessibilityLabel;
    }

    public boolean isTouchUnresponsive() {
        return mTouchUnresponsive;
    }

    /**
     * Builder class for a Tap Event.
     */
    public static class TapEventBuilder extends Builder<TapEvent> {

        @NonNull
        private String mTouchPath;
        @Nullable
        private String mViewLabel;
        @Nullable
        private String mViewId;
        @Nullable
        private String mViewAccessibiltyLabel;
        private boolean mTouchUnresponsive;


        /**
         * Constructs an Event of the type Tap.
         */
        public TapEventBuilder() {
            super();
            setEventAction(TAP);
        }

        //CHECKSTYLE:OFF
        @NonNull
        public String getTouchPath() {
            return mTouchPath;
        }

        @NonNull
        public TapEventBuilder setTouchPath(@NonNull String touchPath) {
            mTouchPath = touchPath;
            return this;
        }

        @Nullable
        public String getViewLabel() {
            return mViewLabel;
        }

        @NonNull
        public TapEventBuilder setViewLabel(@Nullable String viewLabel) {
            mViewLabel = viewLabel;
            return this;
        }

        @Nullable
        public String getViewId() {
            return mViewId;
        }

        @NonNull
        public TapEventBuilder setViewId(@Nullable String viewId) {
            mViewId = viewId;
            return this;
        }

        @Nullable
        public String getViewAccessibiltyLabel() {
            return mViewAccessibiltyLabel;
        }

        @NonNull
        public TapEventBuilder setViewAccessibilityLabel(@Nullable String viewAccessibiltyLabel) {
            mViewAccessibiltyLabel = viewAccessibiltyLabel;
            return this;
        }

        public boolean isTouchUnresponsive() {
            return mTouchUnresponsive;
        }

        public TapEventBuilder setTouchUnresponsive(boolean touchUnresponsive) {
            mTouchUnresponsive = touchUnresponsive;
            return this;
        }
        //CHECKSTYLE:ON

        @Override
        @NonNull
        public TapEvent build() {
            return new TapEvent(this);
        }

    }
}