package com.contentsquare.android.internal.network;


import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.async.ThreadExecutor;
import com.contentsquare.android.internal.async.TransactionRunnable;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.HttpConnection;

import org.json.JSONObject;

import javax.inject.Inject;

/**
 * A Network Transaction is to be used in order to send data to a server.
 * There is no other logic here than that which will allow transactions to be executed
 * in a pool of threads, so as to be as efficient as possible.
 */
public class NetworkTransaction {
    @Inject
    ThreadExecutor mExecutor;

    private final Logger mLogger = new Logger("NetworkTransaction");

    /**
     * Constructs a NetworkTransaction.
     */
    public NetworkTransaction() {
        SingletonProvider.getSessionComponent().inject(this);
    }

    /**
     * Schedules the sending of a json file.
     *
     * @param json     the jsonObject to be sent
     * @param endpoint the url where to send the json
     * @param listener a listener that will get callbacks for successful
     *                 and failed attempts to send the json
     */
    public void scheduleSend(@NonNull final JSONObject json, @NonNull final String endpoint,
            @NonNull final NetworkTransactionListener listener) {
        Runnable runnable = new TransactionRunnable() {
            @Override
            public void rollBack() {
                listener.fail();
            }

            @Override
            public void run() {
                HttpConnection.HttpResponse result =
                        getHttpConnection().performPostWithJson(
                                endpoint,
                                json);

                if (result.positive()) {
                    mLogger.i("Sending json to URL: %s", endpoint);
                    listener.pass();
                    return;
                }

                mLogger.i("Failed to send json to URL: %s", endpoint);
                listener.fail();
            }
        };

        boolean dispatchSuccessful = mExecutor.dispatch(runnable);

        if (!dispatchSuccessful) {
            mLogger.w("dispatch not successful, failing!");
            listener.fail();
        } else {
            mLogger.w("dispatch success, should exec soon!");
        }
    }

    @VisibleForTesting
    HttpConnection getHttpConnection() {
        return new HttpConnection();
    }


    /**
     * An interface that must be implemented in order to get the responses
     * from the transaction's outcome.
     */
    public interface NetworkTransactionListener {
        /**
         * This method will be called when a NetworkTransaction succeeds.
         */
        void pass();

        /**
         * This method will be called when a NetworkTransaction fails.
         */
        void fail();
    }
}