package com.contentsquare.android.internal.preferences;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.util.MD5Hash;
import com.contentsquare.android.internal.util.Strings;


/**
 * Preferences helper implements only common methods for accessing data in any kind of shared
 * preferences. This helper will store any common type, using a md5 hash of the label to get some
 * obscurity.
 */
public class PrefsHelper {

    private static final String CS_PREFS_NAME = "cs";

    /**
     * The Prefs KEY used to persist the last computed sampling interval segment.
     */
    public static final String LAST_SEGMENT = "last_segment";

    /**
     * The Prefs KEY used to persist the last computed boolean value which should tell if the
     * API should be enabled for this device or not.
     */
    public static final String TRACKABLE = "trackable";

    @VisibleForTesting
    final SharedPreferences mSharedPreferences;

    /**
     * Preferences Helper class, which allows ease of use of local preferences.
     *
     * @param context a valid application context.
     */
    public PrefsHelper(@NonNull Context context) {
        mSharedPreferences = context.getSharedPreferences(CS_PREFS_NAME, Context.MODE_PRIVATE);
    }

    /**
     * Check if a property exists in the preferences.
     *
     * @param label - a non-null string with the name of interest
     * @return true if it exists, false if it doesn't, or label is null
     */
    public boolean containsProperty(@NonNull String label) {
        return !Strings.isNullOrEmpty(label) && mSharedPreferences.contains(getHashedLabel(label));
    }

    /**
     * Store a boolean property.
     *
     * @param label - the label
     * @param value - the value
     */
    public void putBoolean(@NonNull String label, boolean value) {
        if (Strings.isNullOrEmpty(label)) {
            return;
        }
        SharedPreferences.Editor edit = mSharedPreferences.edit();
        edit.putBoolean(getHashedLabel(label), value);
        edit.apply();
    }

    /**
     * Get a boolean property.
     *
     * @param label - the label
     * @return the value, or false (default value == false)
     */
    public boolean getBoolean(@NonNull String label, boolean defaultValue) {
        if (Strings.isNullOrEmpty(label)) {
            return defaultValue;
        }
        return mSharedPreferences.getBoolean(getHashedLabel(label), defaultValue);
    }

    /**
     * Store an int property.
     *
     * @param label - the label
     * @param value - the value
     */
    public void putInt(@NonNull String label, int value) {
        if (Strings.isNullOrEmpty(label)) {
            return;
        }
        SharedPreferences.Editor edit = mSharedPreferences.edit();
        edit.putInt(getHashedLabel(label), value);
        edit.apply();
    }

    /**
     * Get a int property.
     *
     * @param label - the label
     * @return the value
     */
    public int getInt(@NonNull String label, int defaultValue) {
        if (Strings.isNullOrEmpty(label)) {
            return defaultValue;
        }
        return mSharedPreferences.getInt(getHashedLabel(label), defaultValue);
    }

    /**
     * Store a float property.
     *
     * @param label - the label
     * @param value - the value
     */
    public void putFloat(@NonNull String label, float value) {
        if (Strings.isNullOrEmpty(label)) {
            return;
        }
        SharedPreferences.Editor edit = mSharedPreferences.edit();
        edit.putFloat(getHashedLabel(label), value);
        edit.apply();
    }

    /**
     * Get a float property.
     *
     * @param label        - the label
     * @param defaultValue default return in the case there's no such key
     * @return the value
     */
    public float getFloat(@NonNull String label, float defaultValue) {
        if (Strings.isNullOrEmpty(label)) {
            return defaultValue;
        }
        return mSharedPreferences.getFloat(getHashedLabel(label), defaultValue);
    }

    /**
     * Stores a string in the preferences.
     *
     * @param label - the label
     * @param value - the value
     */
    public void putString(@NonNull String label, @Nullable String value) {
        if (Strings.isNullOrEmpty(label)) {
            return;
        }

        SharedPreferences.Editor edit = mSharedPreferences.edit();
        edit.putString(getHashedLabel(label), value);
        edit.apply();
    }

    /**
     * Reads a string from properties.
     *
     * @param label        the label under which it was stored.
     * @param defaultValue the default value, in the case there is no value in the preferences.
     * @return the value stored or the defaultValue provided
     */
    @Nullable
    public String getString(@NonNull String label, @Nullable String defaultValue) {
        if (Strings.isNullOrEmpty(label)) {
            return defaultValue;
        }

        return mSharedPreferences.getString(getHashedLabel(label), defaultValue);
    }

    /**
     * Cleans the SharedPreferences.
     */
    public void deleteAndResetAll() {
        SharedPreferences.Editor edit = mSharedPreferences.edit();
        edit.clear();
        edit.apply();
    }

    /**
     * gets a hashed label for a property.
     *
     * @param label the label which is to be hashed.
     * @return the hashed value
     */
    @Nullable
    private String getHashedLabel(String label) {
        return new MD5Hash(label).getMd5Hash();
    }
}
