package com.contentsquare.android.internal.screengraph;

import android.support.annotation.Nullable;

import com.contentsquare.android.internal.util.ViewUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

/**
 * Data representation of a serializable View descriptor.
 */
public class JsonView {

    private static final String ID = "id";
    private static final String HEIGHT = "height";
    private static final String WIDTH = "width";
    private static final String CHILDREN = "views";
    private static final String X = "x";
    private static final String Y = "y";
    private static final String BITMAP = "bmp";
    private static final String BACKGROUND = "bg";
    private static final String METADATA = "metadata";
    private static final String VISIBILITY = "visibility";

    private String mId;
    private int mWidth;
    private int mHeight;
    private int mPosX;
    private int mPosY;
    private JSONObject mMetadata;
    private List<JsonView> mChildren;
    private byte[] mBitmap;
    private String mBackground;
    private boolean mVisible;

    private final ViewUtil mViewUtil;


    /**
     * Build a JsonView.
     *
     * @param viewUtil a viewUtil - do an injection using SingletonProvider
     */
    public JsonView(ViewUtil viewUtil) {
        mViewUtil = viewUtil;
    }

    public String getId() {
        return mId;
    }

    public void setId(String id) {
        mId = id;
    }

    public int getWidth() {
        return mWidth;
    }

    public void setWidth(int width) {
        mWidth = width;
    }

    public int getHeight() {
        return mHeight;
    }

    public void setHeight(int height) {
        mHeight = height;
    }

    public List<JsonView> getChildren() {
        return mChildren;
    }

    public void setChildren(List<JsonView> children) {
        mChildren = children;
    }

    int getPosX() {
        return mPosX;
    }

    public void setPosX(int posX) {
        mPosX = posX;
    }

    int getPosY() {
        return mPosY;
    }

    public void setPosY(int posY) {
        mPosY = posY;
    }

    JSONObject getMetadata() {
        return mMetadata;
    }

    public void setMetadata(JSONObject metadata) {
        mMetadata = metadata;
    }

    @Nullable
    public String getBackground() {
        return mBackground;
    }

    public void setBackground(String background) {
        mBackground = background;
    }

    @SuppressFBWarnings("EI_EXPOSE_REP")
    @Nullable
    public byte[] getBitmap() {
        return mBitmap;
    }

    @SuppressFBWarnings("EI_EXPOSE_REP2")
    public void setBitmap(byte[] bitmap) {
        mBitmap = bitmap;
    }

    public boolean isVisible() {
        return mVisible;
    }

    public void setVisible(boolean visible) {
        mVisible = visible;
    }

    /**
     * Returns the value mapped corresponding to the current object if it exists and is a {@link
     * JSONObject}, or null otherwise.
     *
     * @return {@link JSONObject} with the values corresponding to the current object.
     */
    @Nullable
    JSONObject toJson() throws JSONException {

        JSONObject jsonObject = new JSONObject();
        JSONArray jsonArray = new JSONArray();

        jsonObject.put(ID, getId());
        jsonObject.put(HEIGHT, getHeight());
        jsonObject.put(WIDTH, getWidth());
        jsonObject.put(X, getPosX());
        jsonObject.put(Y, getPosY());
        if (mBitmap != null) {
            jsonObject.put(BITMAP, mViewUtil.encodeImage(mBitmap));
        }
        if (mBackground != null) {
            jsonObject.put(BACKGROUND, mBackground);
        }
        jsonObject.put(VISIBILITY, isVisible());
        jsonObject.put(METADATA, getMetadata());
        if (getChildren() != null) {
            for (JsonView obj : getChildren()) {
                jsonArray.put(obj.toJson());
            }
            jsonObject.put(CHILDREN, jsonArray);
        }

        return jsonObject;
    }
}