package com.contentsquare.android.internal.screengraph;

import static com.contentsquare.android.internal.util.ResourceUtils.getResourceEntryName;

import android.app.Activity;
import android.app.Application;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AppCompatActivity;
import android.view.View;
import android.view.ViewGroup;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.ViewUtil;


/**
 * A generator of Screen Descriptors a.k.a.
 * A Screen URL is a "web-like" URI which describes a screen.
 * " protocol://app-identifier/activity/container?query-parameters "
 *
 * @see <a href="https://contentsquare.atlassian.net/wiki/spaces/RD/pages/54329418/URL+Definition">
 * Confluence Specs</a>
 */
public class PathGenerator {

    // Query Params Keys
    /**
     * The key for the url title parameter.
     */
    public static final String SCREEN_TITLE_KEY = "title";
    private static final String PROTOCOL = "app-and://";
    @NonNull
    private final Application mApplication;
    @NonNull
    private final ViewUtil mViewUtil;
    private final Logger mLogger = new Logger("PathGenerator");

    /**
     * Constructs an object which can generate a screen graph.
     */
    public PathGenerator(@NonNull Application application,
                         @NonNull ViewUtil viewUtil) {
        mApplication = application;
        mViewUtil = viewUtil;
    }

    /**
     * Generate a URL description for a screen.
     *
     * @param rootView the root view for this screen ( the decor view ).
     * @return a URL descriptor for this view, or null if we're not actively listening to an
     * activity (hence it's null).
     */
    @NonNull
    public String generateUrl(@NonNull Activity activity, @NonNull final ViewGroup rootView) {

        Uri rootUri = getRootUri();
        final String url = getPagePath(rootUri, rootView, activity);

        mLogger.d("Complete Path: %s", url);
        return url;
    }

    /**
     * Generates an URL description for a sub - screen.
     *
     * @param activity  as the parent {@link Activity}
     * @param fragment  as the {@link Fragment} which was resumed.
     * @param viewGroup as the root {@link ViewGroup}.
     * @return an URL descriptor for this View.
     */
    @NonNull
    public String generateUrl(@NonNull final Activity activity,
                              @NonNull final Fragment fragment,
                              @Nullable final ViewGroup viewGroup) {
        final Uri rootUri = getRootUri();
        return getSubPagePath(rootUri, activity, fragment, viewGroup);

    }

    @NonNull
    private String getPagePath(@NonNull Uri root,
                               @NonNull ViewGroup rootView,
                               @NonNull Activity activity) {
        final String pagePath = activity.getClass().getSimpleName().replace("Activity", "");
        View biggestView = mViewUtil.getBiggestViewInHierarchy(rootView);
        final String viewPath = getResourceEntryName(biggestView,
                "id_" + biggestView.getClass().getSimpleName());


        final Uri.Builder ub = root.buildUpon();
        ub.appendPath(pagePath).appendPath(viewPath);

        addQueryParams(ub, activity);
        return ub.toString();
    }

    @NonNull
    private String getSubPagePath(@NonNull Uri root,
                                  @NonNull Activity activity,
                                  @NonNull Fragment fragment,
                                  @Nullable ViewGroup rootView) {
        final String pagePath = activity.getClass().getSimpleName().replace("Activity", "");
        final String subPagePath = fragment.getClass().getSimpleName().replace("Fragment", "");
        final Uri.Builder ub = root.buildUpon();
        if (rootView != null) {
            View biggestView = mViewUtil.getBiggestViewInHierarchy(rootView);
            final String viewPath = getResourceEntryName(biggestView,
                    "id_" + biggestView.getClass().getSimpleName());
            ub.appendPath(pagePath).appendPath(subPagePath + "_" + viewPath);
            addQueryParams(ub, activity);

        } else {
            ub.appendPath(pagePath).appendPath(subPagePath);
            addQueryParams(ub, activity);
        }

        return ub.toString();
    }

    private void addQueryParams(@NonNull Uri.Builder ub,
                                @NonNull Activity activity) {
        final CharSequence title = getActivityTitle(activity);
        if (title != null && title.length() > 0) {
            ub.appendQueryParameter(SCREEN_TITLE_KEY, String.valueOf(title));
        }
    }

    @Nullable
    private CharSequence getActivityTitle(@NonNull Activity activity) {
        CharSequence title = null;
        final android.app.ActionBar actionBar = activity.getActionBar();
        if (actionBar != null) {
            title = actionBar.getTitle();
        }
        if (activity instanceof AppCompatActivity) {
            final ActionBar supportActionBar = ((AppCompatActivity) activity).getSupportActionBar();
            if (supportActionBar != null) {
                title = supportActionBar.getTitle();
            }
        }
        if (title == null || title.length() <= 0) {
            title = activity.getTitle();
        }
        return title;
    }

    @NonNull
    private Uri getRootUri() {
        final String appId = mApplication.getPackageName();
        return Uri.parse(PROTOCOL + appId);
    }
}
