package com.contentsquare.android.internal.screengraph;


import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.view.ViewGroup;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.ViewUtil;

import org.json.JSONException;
import org.json.JSONObject;

import hugo.weaving.DebugLog;

/**
 * A simple generator of screenshots.
 */
public class ScreenshotGenerator {

    @VisibleForTesting
    static final String URL = "url";
    @VisibleForTesting
    static final String PNG = "png";
    @VisibleForTesting
    static final String HASH_ID = "hash_id";

    private final ViewUtil mViewUtil;

    /**
     * ScreenshotGenerator constructor.
     *
     * @param viewUtil should be injected via SingletonProvider
     */
    public ScreenshotGenerator(@NonNull ViewUtil viewUtil) {
        mViewUtil = viewUtil;
    }

    /**
     * Generate a JSONObject containing the serialized screenshot and the current screen's url.
     *
     * @param rootView the root view for the screenshot that will be returned in the JSON
     * @param hashId   as {@code int} - unique id based on the hashcode of touch target path
     * @return a JSON containing a serialized screenshot of the viewtree passed in parameter
     * and the screen's URL
     */
    @DebugLog
    @Nullable
    public JSONObject obtain(@Nullable String url, @NonNull ViewGroup rootView, int hashId) {
        // TODO: 10/31/17 Move the initialization of this logger in the constructor and use only
        // one instance in this class.
        final Logger mLogger = new Logger("ScreenshotGenerator");

        String serializedScreenshot = getSerializedScreenshot(rootView);
        JSONObject object = new JSONObject();

        try {
            object.put(URL, url);
            object.put(PNG, serializedScreenshot);
            object.put(HASH_ID, hashId);
        } catch (JSONException e) {
            mLogger.w(e, "Failed to create the screenshot's JSON");
            return null;
        }

        return object;
    }

    /**
     * Returns a serialized bitmap as a String from the rootView it is given.
     *
     * @param rootView the root view used to get the screenshot
     * @return a String that is a serialized version of the screenshot
     */
    @VisibleForTesting
    @NonNull
    String getSerializedScreenshot(@NonNull ViewGroup rootView) {
        byte[] screenshotByteArray = mViewUtil.toByteArray(rootView);
        return mViewUtil.encodeImage(screenshotByteArray);
    }
}