package com.contentsquare.android.internal.screengraph;

import static com.contentsquare.android.internal.DeviceInfo.HEIGHT;
import static com.contentsquare.android.internal.DeviceInfo.WIDTH;
import static com.contentsquare.android.internal.dagger.SingletonProvider.getSessionComponent;
import static com.contentsquare.android.internal.util.ResourceUtils.NULL_STRING_ID;
import static com.contentsquare.android.internal.util.ResourceUtils.getResourceEntryName;

import android.app.Application;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.view.View;
import android.view.ViewGroup;

import com.contentsquare.android.internal.DeviceInfo;
import com.contentsquare.android.internal.ui.glasspane.GlassPaneLayout;
import com.contentsquare.android.internal.util.ViewUtil;

import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

/**
 * A implementation of the visitor pattern which traverses a Tree View composed of {@link
 * android.view.ViewGroup}
 * that executes actions on every {@link View} or {@link ViewGroup} it lands on.
 */
public class TreeTraverser {
    /**
     * Glass pane (top and bottom) resource id names. the top has a "_top" appendix.
     */
    public static final String CS_GLASS = "cs_glass_pane";

    /**
     * Top Glass pane resource id name.
     */
    public static final String CS_GLASS_TOP = "cs_glass_pane_top";

    /**
     * {@link com.contentsquare.android.internal.ui.glasspane.GlassPaneLayout} class name, used for
     * avoiding it in graphs and path segments.
     */
    public static final String CS_LAYOUT_NAME = "GlassPaneLayout";
    /**
     * The common name for the GlassPane, to be used in resources.
     */
    public static final String CS_GLASS_CONTAINER = "glass_pane";
    private static final String GRAPH_VERSION = "1";
    @Inject
    DeviceInfo mDeviceInfo;
    @Inject
    Application mApplication;
    @Inject
    ViewUtil mViewUtil;

    /**
     * Constructs a Tree Traverser.
     */
    TreeTraverser() {
        getSessionComponent().inject(this);
    }

    /**
     * Algorithm that traverse the Tree View while building a JSONObject containing all the node.
     *
     * @param view   a ViewGroup node containing children.
     * @param url    as {@link String} the url of the current ViewGroup
     * @param hashId as {@code int} - unique id based on the hashcode of touch target path
     * @return a JSONObject containing all the nodes and the hierarchical view of a tree
     */
    @NonNull
    Screen traverse(@NonNull ViewGroup view, @Nullable String url, int hashId) {
        Screen screenTreeGraph = new Screen();
        screenTreeGraph.setHeight(mDeviceInfo.getDeviceResolution().get(HEIGHT));
        screenTreeGraph.setWidth(mDeviceInfo.getDeviceResolution().get(WIDTH));
        screenTreeGraph.setVersion(mDeviceInfo.getSdkVersion());
        screenTreeGraph.setVersionJson(GRAPH_VERSION);
        screenTreeGraph.setHashId(hashId);
        screenTreeGraph.setVersionApp(mDeviceInfo.getApplicationVersion());

        List<JsonView> listChild = new ArrayList<>();
        listChild.add(traverseView(view));

        screenTreeGraph.setChildren(listChild);
        screenTreeGraph.setUrl(url);

        return screenTreeGraph;
    }

    @VisibleForTesting
    JsonView traverseView(ViewGroup view) {
        if (view instanceof GlassPaneLayout) {
            // if this is the GlassPaneLayout, skip by redirecting to the child,
            // not processing this view
            return traverseView((ViewGroup) view.getChildAt(0));
        }
        JsonView viewObj = mViewUtil.toObject(view);
        List<JsonView> listChildrenView = new ArrayList<>();
        final int childCount = view.getChildCount();

        for (int i = 0; i < childCount; ++i) {
            final View child = view.getChildAt(i);
            String childResource = getResourceEntryName(child, NULL_STRING_ID);
            if (childResource.startsWith(CS_GLASS_TOP)) {
                // if we run by the cs_glass_pane_top view, just skip it
                continue;
            }
            if (child instanceof ViewGroup) {
                listChildrenView.add(traverseView((ViewGroup) child));
            } else {
                listChildrenView.add(mViewUtil.toObject(child));
            }
        }

        viewObj.setChildren(listChildrenView);
        return viewObj;
    }
}
