package com.contentsquare.android.internal.screenmonitoring;

import android.app.Activity;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;

import com.contentsquare.android.internal.logging.Logger;

/**
 * Contains different validators according with the Activity type. Based on the validation result
 * we are able to apply the right strategy to handle the Activity monitor.
 */

class ActivityValidator {

    /**
     * The Log Tag for this class.
     */
    public static final String TAG = "ActivityValidator";

    @NonNull
    private final Logger mLogger;

    /**
     * Creates a new instance of {@link ActivityValidator}.
     *
     * @param logger as {@link Logger}
     */
    ActivityValidator(@NonNull Logger logger) {
        this.mLogger = logger;
    }

    /**
     * Checks is the current activity is a valida {@link FragmentActivity}
     *
     * @param activity as a {@link Activity}
     * @return true if this activity is suitable for a strategy.
     */
    boolean isValidFragmentActivity(@NonNull Activity activity) {
        if (FragmentActivity.class.isAssignableFrom(activity.getClass())) {
            try {
                final String classNameToCheck =
                        "android.support.v4.app.FragmentManager$FragmentLifecycleCallbacks";
                checkClassForName(classNameToCheck);
                final FragmentManager supportFragmentManager = ((FragmentActivity) activity)
                        .getSupportFragmentManager();
                checkMethodOnFragmentManager(supportFragmentManager);

                return true;

            } catch (ClassNotFoundException e) {
                mLogger.w("Could not attach to Activity due to wrong Support API version.", e);

            } catch (NoSuchMethodException e1) {
                mLogger.w("Could not attach to Activity due to wrong Support API version.", e1);
            }
        }

        return false;
    }

    @VisibleForTesting
    Class<?> checkClassForName(@NonNull String name) throws ClassNotFoundException {
        return Class.forName(name);
    }

    @VisibleForTesting
    void checkMethodOnFragmentManager(@NonNull FragmentManager manager)
            throws NoSuchMethodException {
        manager.getClass()
                .getDeclaredMethod("registerFragmentLifecycleCallbacks",
                        FragmentManager.FragmentLifecycleCallbacks.class, boolean.class);

    }

}
