package com.contentsquare.android.internal.screenmonitoring;

import android.app.Activity;
import android.support.annotation.CheckResult;
import android.support.annotation.NonNull;
import android.support.annotation.RestrictTo;

import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.screenmonitoring.strategies.IActivityMonitoringStrategy;
import com.contentsquare.android.internal.util.DateTimeUtil;

import java.util.Map;


/**
 * The implementation of the {@link IScreenMonitoringService} interface.
 */

public class ScreenMonitoringService implements IScreenMonitoringService {

    /**
     * The Log Tag for this class.
     */
    public static final String TAG = "ScreenMonitoringService";


    @NonNull
    private final StrategiesCache mStrategiesCache;

    /**
     * Creates a new instance of this class.
     *
     * @param activityValidator as a {@link ActivityValidator}
     */
    ScreenMonitoringService(@NonNull ActivityValidator activityValidator,
                            @NonNull DateTimeUtil dateTimeUtil) {
        this.mStrategiesCache = new StrategiesCache(activityValidator, dateTimeUtil);
    }

    /**
     * Creates a new instance of this class.
     */
    ScreenMonitoringService() {
        this(new ActivityValidator(new Logger(ActivityValidator.TAG)),
                SingletonProvider.getAppComponent().getDateTimeUtil());
    }

    @Override
    @CheckResult
    public boolean attachTo(@NonNull Activity activity, IScreenChangedCallback callback) {

        final IActivityMonitoringStrategy strategy = mStrategiesCache.get(activity);
        return strategy.attachTo(activity, callback);
    }

    @Override
    public void detachFrom(@NonNull Activity activity) {
        final IActivityMonitoringStrategy strategy = mStrategiesCache.getFromCache(activity);
        if (strategy != null) {
            strategy.detachFrom(activity);
        }
    }

    @RestrictTo(RestrictTo.Scope.TESTS)
    @NonNull
    Map<Class<?>, IActivityMonitoringStrategy> strategies() {
        return mStrategiesCache.strategies();
    }

}
