package com.contentsquare.android.internal.screenmonitoring;

import android.app.Activity;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RestrictTo;
import android.support.v4.util.ArrayMap;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.screenmonitoring.strategies.DummyStrategy;
import com.contentsquare.android.internal.screenmonitoring.strategies.FragmentActivityMonitoringStrategy;
import com.contentsquare.android.internal.screenmonitoring.strategies.IActivityMonitoringStrategy;
import com.contentsquare.android.internal.util.DateTimeUtil;

import java.util.Collections;
import java.util.Map;

/**
 * A pool of {@link IActivityMonitoringStrategy} which builds and re - uses a new strategy for
 * each provided {@link Activity} type.
 */

class StrategiesCache {

    private static final String TAG = "StrategiesCache";

    private final ArrayMap<Class<?>, IActivityMonitoringStrategy> mStrategies
            = new ArrayMap<>();

    @NonNull
    private final ActivityValidator mActivityValidator;

    @NonNull
    private final Logger mLogger = new Logger(TAG);

    @NonNull
    private final DateTimeUtil mDateTimeUtils;

    /**
     * Creates a new instance of this class.
     *
     * @param activityValidator as {@link ActivityValidator}
     * @param dateTimeUtils    as {@link DateTimeUtil}
     */
    StrategiesCache(@NonNull ActivityValidator activityValidator,
                    @NonNull DateTimeUtil dateTimeUtils) {
        this.mActivityValidator = activityValidator;
        this.mDateTimeUtils = dateTimeUtils;
    }

    /**
     * Builds a new {@link IActivityMonitoringStrategy} and caches this instance in order to be
     * re-used during the entire session for the specific {@link Activity} type.
     *
     * @param activity as the {@link Activity}
     * @return a new {@link IActivityMonitoringStrategy} which can monitor the provide
     * {@link Activity} type.
     */
    IActivityMonitoringStrategy get(@NonNull Activity activity) {
        return determineAndProvideStrategy(activity);
    }


    /**
     * Checks the cache and returns the mapped {@link IActivityMonitoringStrategy} for the provided
     * activity.
     *
     * @param activity as the {@link Activity} for which we requested
     *                 the cached {@link IActivityMonitoringStrategy}
     * @return the related {@link IActivityMonitoringStrategy} or null
     */
    @Nullable
    IActivityMonitoringStrategy getFromCache(@NonNull Activity activity) {
        return mStrategies.get(activity.getClass());
    }


    /**
     * Returns an immutable Map from the existing strategies cache.
     * Only meant for testing purposes.
     *
     * @return the strategies cache status
     * as an immutable {@code Map<Class<?>,IActivityMonitoringStrategy>}
     */
    @RestrictTo(RestrictTo.Scope.TESTS)
    Map<Class<?>, IActivityMonitoringStrategy> strategies() {
        return Collections.unmodifiableMap(mStrategies);
    }

    @NonNull
    private IActivityMonitoringStrategy determineAndProvideStrategy(@NonNull Activity
                                                                            activity) {
        if (mActivityValidator.isValidFragmentActivity(activity)) {
            IActivityMonitoringStrategy strategy =
                    mStrategies.get(activity.getClass());

            if (strategy == null) {
                strategy = new FragmentActivityMonitoringStrategy(mDateTimeUtils);
                mStrategies.put(activity.getClass(), strategy);
            }
            mLogger.d("[ DetermineAndProvideStrategy ]: Strategy found for the FragmentActivity "
                    + "with the right Support API.");
            // attach the callback
            return strategy;
        }


        return new DummyStrategy();
    }

}
