package com.contentsquare.android.internal.screenmonitoring.strategies;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RestrictTo;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;
import android.support.v4.util.ArrayMap;
import android.support.v4.util.SparseArrayCompat;

import com.contentsquare.android.internal.screenmonitoring.IScreenChangedCallback;
import com.contentsquare.android.internal.util.DateTimeUtil;
import com.contentsquare.android.internal.util.MathUtil;

import java.util.Map;

/**
 * Implementation of the {@link IActivityMonitoringStrategy} which will be applied in case the
 * passed activity support the callback on the Fragment Manager.
 */

public class FragmentActivityMonitoringStrategy implements IActivityMonitoringStrategy {

    private final Map<Activity, ScreenChangedCallbackWrapper> mCallbacks = new ArrayMap<>();

    @NonNull
    private final DateTimeUtil mDateTimeUtils;

    /**
     * Creates a new instance of this class.
     *
     * @param dateTimeUtil as {@link DateTimeUtil}
     */
    public FragmentActivityMonitoringStrategy(@NonNull DateTimeUtil dateTimeUtil) {
        this.mDateTimeUtils = dateTimeUtil;
    }

    @Override
    public boolean attachTo(@NonNull Activity activity, @NonNull IScreenChangedCallback callback) {
        if (!(activity instanceof FragmentActivity)) {
            return false;
        }
        final ScreenChangedCallbackWrapper callbackWrapper =
                new ScreenChangedCallbackWrapper(mDateTimeUtils, callback);
        mCallbacks.put(activity, callbackWrapper);
        ((FragmentActivity) activity).getSupportFragmentManager()
                .registerFragmentLifecycleCallbacks(callbackWrapper, true);
        return true;
    }


    @Override
    public void detachFrom(@NonNull Activity activity) {
        ((FragmentActivity) activity).getSupportFragmentManager()
                .unregisterFragmentLifecycleCallbacks(mCallbacks.remove(activity));
    }

    @RestrictTo(RestrictTo.Scope.TESTS)
    @Override
    @Nullable
    public IScreenChangedCallback callback(@NonNull Activity activity) {
        final ScreenChangedCallbackWrapper wrapper = mCallbacks.get(activity);
        return wrapper != null ? wrapper.wrappedCallback() : null;
    }

    @RestrictTo(RestrictTo.Scope.TESTS)
    @Nullable
    ScreenChangedCallbackWrapper unwrappedCallback(@NonNull Activity activity) {
        return mCallbacks.get(activity);
    }

    static class ScreenChangedCallbackWrapper
            extends FragmentManager.FragmentLifecycleCallbacks
            implements IScreenChangedCallback {

        private final SparseArrayCompat<Long> mLoadingTimeMapper = new SparseArrayCompat<>();
        private final DateTimeUtil mDateTimeUtils;


        @NonNull
        private final IScreenChangedCallback mScreenChangedCallback;

        ScreenChangedCallbackWrapper(@NonNull DateTimeUtil dateTimeUtils,
                                     @NonNull IScreenChangedCallback screenChangedCallback) {
            this.mDateTimeUtils = dateTimeUtils;
            this.mScreenChangedCallback = screenChangedCallback;
        }

        @Override
        public void onFragmentPreAttached(FragmentManager fm, Fragment fragment, Context context) {
            super.onFragmentPreAttached(fm, fragment, context);
            mLoadingTimeMapper.put(fragment.hashCode(), mDateTimeUtils.currentTimeMillis());
        }

        @Override
        public void onFragmentStarted(FragmentManager fm, Fragment fragment) {
            // check if we already have an entry for this fragment in the timer mapper.
            // If not this means this fragment resumes from backstack so we will have to take
            // this event as a start time reference
            final int key = mLoadingTimeMapper.indexOfKey(fragment.hashCode());
            if (key < 0) {
                mLoadingTimeMapper.put(fragment.hashCode(), mDateTimeUtils.currentTimeMillis());
            }
            super.onFragmentStarted(fm, fragment);
        }

        @Override
        public void onFragmentResumed(FragmentManager fm, Fragment fragment) {
            super.onFragmentResumed(fm, fragment);
            final long currentTimeMillis = mDateTimeUtils.currentTimeMillis();
            final int key = fragment.hashCode();
            final long attachedTime = mLoadingTimeMapper.get(key, currentTimeMillis);
            final long loadingTime = currentTimeMillis - attachedTime;
            mLoadingTimeMapper.remove(key);
            onScreenChanged(fragment.getActivity(), fragment, MathUtil.safeLongToInt(loadingTime));
        }

        @Override
        public void onScreenChanged(@NonNull Activity activity,
                                    @NonNull Fragment fragment,
                                    int loadingTime) {
            mScreenChangedCallback.onScreenChanged(activity, fragment, loadingTime);
        }

        @NonNull
        IScreenChangedCallback wrappedCallback() {
            return mScreenChangedCallback;
        }



        @RestrictTo(RestrictTo.Scope.TESTS)
        SparseArrayCompat<Long> loadingTimeMapper() {
            return mLoadingTimeMapper;
        }
    }
}
