package com.contentsquare.android.internal.ui.glasspane;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.Window;
import android.widget.FrameLayout;

import com.contentsquare.android.R;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.util.ViewUtil;

import javax.inject.Inject;

/**
 * The GlassPane is a composit of a View and it's TouchListener. This is the transparent View which
 * is injected in the Activity's view hierarchy on the DecorView
 * level. This is the main touch interceptor.
 * Do not make instances of this class, it's being managed by Dagger.
 */
public class GlassPane implements IGlassPane {
    @NonNull
    private final Logger mLogger = new Logger("GlassPane");
    @NonNull
    private final GlassPaneLayout mGlassPaneLayout;
    @NonNull
    private final View mGlassCoverView;
    @Nullable
    private FrameLayout mDecorView;
    @Nullable
    private String mCurrentScreenUrl;

    /**
     * Injected constructor, used by Dagger.
     * *DO NOT USE, MANAGED BY DAGGER*
     */
    @SuppressLint("InflateParams")
    @Inject
    public GlassPane(@NonNull final Context context) {
        this((GlassPaneLayout) LayoutInflater.from(context).inflate(R.layout.glass_pane, null,
                false), LayoutInflater.from(context).inflate(R.layout.glass_pane_top, null, false));
    }

    /**
     * Constructs a GlassPane from a glassPaneLayout and  a listener.
     * *DO NOT USE, MANAGED BY DAGGER*
     *
     * @param glassPaneLayout the glassPaneLayout which will be injected
     */
    public GlassPane(@NonNull final GlassPaneLayout glassPaneLayout,
                     @NonNull final View glassCoverView) {
        mGlassPaneLayout = glassPaneLayout;
        mGlassCoverView = glassCoverView;
    }

    @Override
    @NonNull
    public final GlassPaneLayout getGlassPaneLayout() {
        return mGlassPaneLayout;
    }

    @Override
    public final void attachGlassPane(@NonNull final Activity liveActivity) {
        mLogger.w("attaching Glass");
        final Window window = liveActivity.getWindow();
        mDecorView = (FrameLayout) window.findViewById(android.R.id.content);
        if (mDecorView == null) {
            // we failed to get content, exiting (happens on Google Ad fullscreen activities)
            return;
        }
        final View view = mDecorView.getChildAt(0);
        if (view == null) {
            // we failed to get the child view, exiting (happens on Google Ad fullscreen activities)
            return;
        }
        final GlassPaneLayout glass = getGlassPaneLayout();
        mDecorView.removeView(view);
        glass.addView(view);
        final ViewUtil viewUtil = SingletonProvider.getAppComponent().getViewUtil();
        viewUtil.verifyViewDetached(glass);
        mDecorView.addView(glass);
        viewUtil.verifyViewDetached(mGlassCoverView);
        mDecorView.addView(mGlassCoverView);
        mGlassCoverView.setOnTouchListener(new View.OnTouchListener() {
            @Override
            @SuppressLint("ClickableViewAccessibility")
            public boolean onTouch(View view, MotionEvent event) {
                // we need this on touch listener JUST to return false.
                // as we don't actually need anything other it's existence.
                return false;
            }
        });
    }

    @Override
    public final void detachGlassPane(@Nullable final Activity liveActivity) {
        mLogger.w("detaching Glass");
        if (liveActivity != null) {
            Window window = liveActivity.getWindow();
            if (window == null) {
                // early exit in case of null decor view (registered occurrence: google ads)
                return;
            }
            FrameLayout decorView = (FrameLayout) window.findViewById(android.R.id.content);
            if (decorView == null) {
                // early exit in case of null decor view (registered occurrence: google ads)
                return;
            }
            restoreDecorViewToDefault(decorView);
        }
        //release the reference to the view so that GC can take over.
        mDecorView = null;
    }

    private void restoreDecorViewToDefault(@NonNull FrameLayout decorView) {
        decorView.setOnHierarchyChangeListener(null);
        GlassPaneLayout glass = decorView.findViewById(R.id.cs_glass_pane);
        View glassTop = decorView.findViewById(R.id.cs_glass_pane_top);
        if (glass != null) {
            View child = glass.getChildAt(0);
            decorView.removeView(glass);
            glass.removeAllViews();
            if (child != null) {
                decorView.addView(child);
            }
        }
        if (glassTop != null) {
            //remove the top of the sandwich
            decorView.removeView(glassTop);
        }
    }

    @Override
    public void stopSession() {
        if (mDecorView != null) {
            mDecorView.setOnHierarchyChangeListener(null);
            pauseListeners();
        }
    }

    @Override
    public void pauseListeners() {
        mGlassPaneLayout.pauseInterception();
    }

    @Override
    public void resumeListeners() {
        mGlassPaneLayout.resumeInterception();
    }

    @Nullable
    @Override
    public FrameLayout getDecorView() {
        return mDecorView;
    }

    @Override
    @Nullable
    public String getCurrentScreenUrl() {
        return mCurrentScreenUrl;
    }

    @Override
    public void setCurrentScreenUrl(@Nullable String currentScreenUrl) {
        mCurrentScreenUrl = currentScreenUrl;
    }
}
