package com.contentsquare.android.internal.ui.glasspane;

import static com.contentsquare.android.internal.dagger.SingletonProvider.getAppComponent;
import static com.contentsquare.android.internal.dagger.SingletonProvider.getSessionComponent;

import android.annotation.TargetApi;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.VelocityTracker;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.LinearLayout;

import com.contentsquare.android.internal.Session;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.model.EventsFactory;
import com.contentsquare.android.internal.model.data.ActionEvent;
import com.contentsquare.android.internal.model.data.ActionEvent.FingerAction;
import com.contentsquare.android.internal.model.data.ActionEvent.FingerDirection;
import com.contentsquare.android.internal.model.data.CrashEvent;
import com.contentsquare.android.internal.model.data.CrashEvent.CrashEventBuilder;
import com.contentsquare.android.internal.model.data.DoubleTapEvent.DoubleTapEventBuilder;
import com.contentsquare.android.internal.model.data.DragEvent.DragEventBuilder;
import com.contentsquare.android.internal.model.data.FlickEvent.FlickEventBuilder;
import com.contentsquare.android.internal.model.data.LongPressEvent.LongPressEventBuilder;
import com.contentsquare.android.internal.model.data.TapEvent.TapEventBuilder;
import com.contentsquare.android.internal.ui.processor.LayoutTraverser;
import com.contentsquare.android.internal.ui.processor.descriptors.PlaneDescriptor;
import com.contentsquare.android.internal.util.DateTimeUtil;
import com.contentsquare.android.internal.util.MathUtil;
import com.contentsquare.android.internal.util.ScreenRecorder;
import com.contentsquare.android.internal.util.ViewUtil;

import java.util.Stack;


/**
 * A parent layout who's only purpose in life is to intercept touch events.
 * See the <a href="{@docRoot}guide/topics/ui/layout/linear.html">Linear Layout</a>
 * guide.
 */
public class GlassPaneLayout extends LinearLayout {

    @VisibleForTesting
    static final int FINGER_THRESHOLD_VELOCITY = 1000; // in dp/s
    private static final int TAP_GESTURE_DISTANCE = 8 * 3; // in dp
    private static final long TAP_GESTURE_TIMEFRAME = 200; // in ms
    private static final int DOUBLE_TAP_TIMEOUT = 200; // in ms
    private static final int TIME_PER_PIXEL = 1000; // in ms
    @VisibleForTesting
    final Handler mHandler = new Handler();
    private final Logger mLog = new Logger("Interceptor");
    @VisibleForTesting
    DateTimeUtil mDateTimeUtil;
    @FingerDirection
    int mFingerDirection = ActionEvent.FINGER_DIRECTION_NULL;
    private boolean mEnableLogs = false;
    private boolean mPaused = false;
    private boolean mProcessingGesture = false;
    private int mTapStartX; // in dp
    private int mTapStartY; // in dp
    private long mGestureStart; // timestamp
    private double mGestureDistance; // in dp
    private long mGestureDuration; // in ms
    private double mGestureVelocity; // in dp/s
    @FingerAction
    private int mGestureType;
    private long mGestureEnd; // timestamp
    private int mTapEndX; // in dp
    private int mTapEndY; // in dp
    private PlaneDescriptor mPlane;
    @Nullable
    private VelocityTracker mVelocityTracker;
    private Session mSession;
    @Nullable
    private ActionEvent mPendingUnhandled;


    /**
     * @see LinearLayout#LinearLayout(Context) .
     */
    public GlassPaneLayout(Context context) {
        super(context);
    }

    /**
     * @see LinearLayout#LinearLayout(Context, AttributeSet) .
     */
    public GlassPaneLayout(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
    }

    /**
     * @see LinearLayout#LinearLayout(Context, AttributeSet, int) .
     */
    public GlassPaneLayout(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    /**
     * @see LinearLayout#LinearLayout(Context, AttributeSet, int, int) .
     */
    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    public GlassPaneLayout(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        mSession = getSessionComponent().getSession();
        mDateTimeUtil = getAppComponent().getDateTimeUtil();
    }

    /**
     * Records all touch events for further processing but follows them untouched to their
     * respective targets.
     *
     * @see LinearLayout#dispatchTouchEvent(MotionEvent)
     */
    @Override
    public boolean dispatchTouchEvent(MotionEvent event) {
        if (!mPaused) {
            processEvent(event);
        } else if (mEnableLogs) {
            mLog.w("Event Interception paused");
        }
        return super.dispatchTouchEvent(event);
    }

    /**
     * Pause the event interception.
     */
    public void pauseInterception() {
        mPaused = true;
    }

    /**
     * Resume the event interception.
     */
    public void resumeInterception() {
        mPaused = false;
    }

    /**
     * Process the events coming from the system and make a call to createScreenEvent send screen
     * events to our event's processor.
     *
     * @param event - the event coming from the system
     */
    private void processEvent(MotionEvent event) {
        final int eventRawX = (int) event.getRawX();
        final int eventRawY = (int) event.getRawY();

        if (mEnableLogs) {
            mLog.d("Found: got event: %s", event.toString());
        }

        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                processActionDown(event);
                initiateVelocityTracker(event);
                return;
            case MotionEvent.ACTION_MOVE:
                if (mVelocityTracker != null) {
                    mVelocityTracker.addMovement(event);
                }
                //if action move or whatever else, cancel the pending.
                mPendingUnhandled = null;
                return;
            case MotionEvent.ACTION_CANCEL:
                reset();
                return;
            case MotionEvent.ACTION_UP:
                int index = event.getActionIndex();
                int pointerId = event.getPointerId(index);

                updateVelocity(pointerId);
                mGestureType = ActionEvent.NO_EVENT;
                processActionUp(eventRawX, eventRawY);
                break;
            default:
                if (mEnableLogs) {
                    mLog.w("Found: unhandled event: %s", event.toString());
                }
                //if action move or whatever else, cancel the pending.
                mPendingUnhandled = null;
                return;
        }

        if (mEnableLogs) {
            mLog.i("Found : mGestureType: %b mGestureDistance: %d mGestureDuration:"
                            + " %l eventRawX: %b eventRawY:%b",
                    mGestureType, mGestureDistance, mGestureDuration, eventRawX, eventRawY);
        }

        if (mGestureType == ActionEvent.NO_EVENT) {
            //ignore bad gesture detection
            return;
        }
        createScreenEvent(mGestureType, mPlane);
    }

    /**
     * Records a possible unhandled event.
     * <p>
     * note: This event would be sent IF we get a consecutive action down.
     */
    private void recordUnhandledTouch() {
        EventsFactory factory = mSession.getEventsFactory();
        boolean unresponsive = true;
        mPendingUnhandled = buildTapEvent(factory,
                mPlane,
                mPlane.getTargetPathDescriptor(),
                unresponsive);
    }

    /**
     * Sends any unhandled event if there is one in storage.
     */
    private void sendUnhandled() {
        mSession.getEventsProcessor().processEvent(
                mSession.getJsonProxy().serializeToJson(mPendingUnhandled));
    }

    private void processActionDown(@NonNull MotionEvent event) {
        int eventRawX = (int) event.getRawX();
        int eventRawY = (int) event.getRawY();
        IGlassPane glass = getSessionComponent().getSession().getGlass();
        FrameLayout decor = glass.getDecorView();
        if (decor == null) {
            mLog.e("decor view is null, exiting...");
            return;
        }
        mPlane = getTouchTarget(glass.getDecorView(), eventRawX, eventRawY);
        if (mProcessingGesture) {
            //if this is the case, means we've received a consecutive action_down.
            // This only happens when we have an "unhandled" touch event.
            sendUnhandled();
            reset();
        }
        recordUnhandledTouch();
        mProcessingGesture = true;

        mTapStartX = ViewUtil.convertPixelsToDps(eventRawX, getContext());
        mTapStartY = ViewUtil.convertPixelsToDps(eventRawY, getContext());

        mGestureStart = mDateTimeUtil.currentTimeMillis();
    }

    private void processActionUp(int eventRawX, int eventRawY) {
        if (mProcessingGesture) {
            mProcessingGesture = false;

            mTapEndX = ViewUtil.convertPixelsToDps(eventRawX, getContext());
            mTapEndY = ViewUtil.convertPixelsToDps(eventRawY, getContext());

            mGestureEnd = mDateTimeUtil.currentTimeMillis();

            mGestureDistance = MathUtil.euclideanDistance(
                    mTapStartX,
                    mTapStartY,
                    mTapEndX,
                    mTapEndY);
            mGestureDuration = mGestureEnd - mGestureStart;

            // if path of finger longer than TAP_GESTURE_DISTANCE => flick or a drag.
            if (mGestureDistance > TAP_GESTURE_DISTANCE) {
                if (mGestureVelocity > FINGER_THRESHOLD_VELOCITY) {
                    mGestureType = ActionEvent.FLICK;
                } else {
                    mGestureType = ActionEvent.DRAG;
                }
                updateFingerDirection(mTapStartX, mTapEndX, mTapStartY, mTapEndY);
            } else {
                //else, if we have a short distance and a short duration, it a tap
                if (mGestureDuration < TAP_GESTURE_TIMEFRAME) {
                    mGestureType = ActionEvent.TAP;
                } else {
                    //last, a short distance and a long duration, it's a long press.
                    mGestureType = ActionEvent.LONG_PRESS;
                }
            }
        } else {
            if (mEnableLogs) {
                mLog.e("Found : We're not processing a gesture? Something is wrong");
            }
            mGestureType = ActionEvent.NO_EVENT;
        }
    }

    /**
     * Traverses the ONLY child of this layout which is the root of the view hierarchy and
     * get the target for the click.
     */
    @SuppressWarnings({"squid:S1149", "squid:S1188"})
    @VisibleForTesting
    @NonNull
    PlaneDescriptor getTouchTarget(@NonNull ViewGroup parent, final int touchX, final int touchY) {
        final Stack<View> targets = new Stack<>();
        //we need to traverse the hierarchy, and push the events to all the views along the way.
        LayoutTraverser.build(new LayoutTraverser.Processor() {
            @Override
            public void process(@NonNull View view) {
                if (checkIfViewFitsTarget(view)
                        && view.getVisibility() == View.VISIBLE) {
                    targets.add(view);
                }
            }

            @Override
            public void process(@NonNull ViewGroup viewGroup) {
                if (checkIfViewFitsTarget(viewGroup)
                        && viewGroup.getVisibility() == View.VISIBLE) {
                    targets.add(viewGroup);
                }
            }

            private boolean checkIfViewFitsTarget(@NonNull View view) {
                int[] location = new int[2];

                view.getLocationOnScreen(location);
                int width = view.getWidth();
                int height = view.getHeight();
                return MathUtil.isPointInPlane(
                        location[0],
                        location[1],
                        width,
                        height,
                        touchX,
                        touchY
                );
            }
        }).traverse(parent);
        // the end of the list

        // as the top is always our glass pane, let's pop it from the stack.
        targets.pop();
        //get the view just after the glass.
        View targetView = targets.pop();

        targets.clear();
        return PlaneDescriptor.getDescriptorFor(targetView);
    }

    @VisibleForTesting
    VelocityTracker getVelocityTracker() {
        return VelocityTracker.obtain();
    }

    @VisibleForTesting
    void createScreenEvent(
            @FingerAction int gestureType,
            @NonNull PlaneDescriptor plane) {

        EventsFactory factory = mSession.getEventsFactory();
        ActionEvent event;
        String targetPathDescriptor;
        switch (gestureType) {
            case ActionEvent.TAP:
                boolean unresponsive = false;
                targetPathDescriptor = plane.getTargetPathDescriptor();
                event = buildTapEvent(factory, plane, targetPathDescriptor, unresponsive);
                // capture the screen
                break;
            case ActionEvent.DOUBLE_TAP:
                targetPathDescriptor = plane.getTargetPathDescriptor();
                event = buildDoubleTapEvent(factory, plane, targetPathDescriptor);
                break;
            case ActionEvent.LONG_PRESS:
                targetPathDescriptor = plane.getTargetPathDescriptor();
                event = buildLongPressEvent(factory, plane, targetPathDescriptor);
                getScreenRecorder()
                        .capture(mSession.getGlass().getDecorView(),
                                mSession.getGlass().getCurrentScreenUrl(),
                                getTargetPathId(targetPathDescriptor));
                break;
            case ActionEvent.DRAG:
                targetPathDescriptor = plane.getTargetPathDescriptor();
                event = buildDragEvent(factory, plane, targetPathDescriptor);
                break;
            case ActionEvent.FLICK:
                targetPathDescriptor = plane.getTargetPathDescriptor();
                event = buildFlickEvent(factory, plane, targetPathDescriptor);
                break;
            case ActionEvent.PINCH:
            case ActionEvent.NO_EVENT:
            default:
                targetPathDescriptor = null;
                CrashEventBuilder crashBuilder = factory.produceEvent(gestureType);
                crashBuilder.setCrashOrigin(CrashEvent.CRASH_SDK);
                crashBuilder.setFatal(false);
                crashBuilder.setMessage("Failed to get event for type: " + gestureType);
                event = crashBuilder.build();
                break;
        }

        mSession.getEventsProcessor().processEvent(
                mSession.getJsonProxy().serializeToJson(event));

    }

    private int getTargetPathId(@Nullable String targetPathDescriptor) {
        return targetPathDescriptor != null ? targetPathDescriptor.hashCode() : 0;
    }

    @NonNull
    @VisibleForTesting
    ScreenRecorder getScreenRecorder() {
        return getSessionComponent().getScreenRecorder();
    }

    @NonNull
    private ActionEvent buildTapEvent(@NonNull EventsFactory factory,
                                      @NonNull PlaneDescriptor plane,
                                      @NonNull String targetPathDescriptor,
                                      boolean unresponsive) {

        TapEventBuilder tapBuilder = factory.produceEvent(ActionEvent.TAP);
        return tapBuilder.setTouchPath(targetPathDescriptor)
                .setViewId(plane.getViewResourceId())
                .setTouchUnresponsive(unresponsive)
                .setViewAccessibilityLabel(plane.getViewAccessibilityLabel())
                .setViewLabel(plane.getViewLabel())
                .build();
    }

    @NonNull
    private ActionEvent buildDoubleTapEvent(@NonNull EventsFactory factory,
                                            @NonNull PlaneDescriptor plane,
                                            @NonNull String targetPathDescriptor) {
        DoubleTapEventBuilder doubleTapEventBuilder =
                factory.produceEvent(ActionEvent.DOUBLE_TAP);
        return doubleTapEventBuilder.setTouchPath(targetPathDescriptor)
                .setViewId(plane.getViewResourceId())
                .setViewAccessibilityLabel(plane.getViewAccessibilityLabel())
                .setViewLabel(plane.getViewLabel())
                .build();
    }

    @NonNull
    private ActionEvent buildLongPressEvent(@NonNull EventsFactory factory,
                                            @NonNull PlaneDescriptor plane,
                                            @NonNull String targetPathDescriptor) {
        LongPressEventBuilder longPressEventBuilder =
                factory.produceEvent(ActionEvent.LONG_PRESS);
        return longPressEventBuilder.setTouchPath(targetPathDescriptor)
                .setViewId(plane.getViewResourceId())
                .setViewAccessibilityLabel(plane.getViewAccessibilityLabel())
                .setViewLabel(plane.getViewLabel())
                .build();
    }

    @NonNull
    private ActionEvent buildDragEvent(@NonNull EventsFactory factory,
                                       @NonNull PlaneDescriptor plane,
                                       @NonNull String targetPathDescriptor) {
        DragEventBuilder dragEventBuilder =
                factory.produceEvent(ActionEvent.DRAG);
        return dragEventBuilder.setTouchPath(targetPathDescriptor)
                .setViewId(plane.getViewResourceId())
                .setViewAccessibilityLabel(plane.getViewAccessibilityLabel())
                .setViewLabel(plane.getViewLabel())
                .setFingerDirection(mFingerDirection)
                .setTargetViewDistanceDragged((int) mGestureDistance)
                .setTargetViewVelocity((int) mGestureVelocity)
                .build();
    }

    @NonNull
    private ActionEvent buildFlickEvent(@NonNull EventsFactory factory,
                                        @NonNull PlaneDescriptor plane,
                                        @NonNull String targetPathDescriptor) {
        FlickEventBuilder flickEventBuilder =
                factory.produceEvent(ActionEvent.FLICK);
        return flickEventBuilder.setTouchPath(targetPathDescriptor)
                .setViewId(plane.getViewResourceId())
                .setViewAccessibilityLabel(plane.getViewAccessibilityLabel())
                .setViewLabel(plane.getViewLabel())
                .setFingerDirection(mFingerDirection)
                .setTargetViewDistanceDragged((int) mGestureDistance)
                .setTargetViewVelocity((int) mGestureVelocity)
                .build();
    }

    private void reset() {
        if (mEnableLogs) {
            mLog.w("Found: resetting state, action cancel or rogue event ");
        }

        mProcessingGesture = false;
        mPendingUnhandled = null;
        mTapStartX = -1;
        mTapStartY = -1;
        mGestureStart = -1;
        mGestureDistance = -1;
        mGestureDuration = -1;
        mGestureVelocity = -1;
        mFingerDirection = ActionEvent.FINGER_DIRECTION_NULL;
        mGestureType = ActionEvent.NO_EVENT;
        mGestureEnd = -1;
        mTapEndX = -1;
        mTapEndY = -1;
        recycleVelocityTracker();
    }

    @VisibleForTesting
    private void updateVelocity(int pointerId) {
        if (mVelocityTracker != null) {
            mVelocityTracker.computeCurrentVelocity(TIME_PER_PIXEL);

            int pxVelocityX = MathUtil.safeFloatToInt(mVelocityTracker.getXVelocity(pointerId));
            int pxVelocityY = MathUtil.safeFloatToInt(mVelocityTracker.getYVelocity(pointerId));

            Context context = getAppComponent().getApplication().getApplicationContext();
            int dpVelocityX = ViewUtil.convertPixelsToDps(pxVelocityX, context);
            int dpVelocityY = ViewUtil.convertPixelsToDps(pxVelocityY, context);

            mGestureVelocity = Math.sqrt(Math.pow(dpVelocityX, 2) + Math.pow(dpVelocityY, 2));
        }
    }

    private void updateFingerDirection(float tapStartX, float tapEndX,
                                       float tapStartY, float tapEndY) {
        float directionX = tapEndX - tapStartX;
        float directionY = tapEndY - tapStartY;

        mFingerDirection = getFingerDirection(directionX, directionY);
    }

    @FingerDirection
    private int getFingerDirection(float directionX, float directionY) {
        @FingerDirection
        int fingerDirection;

        if (Math.abs(directionX) > Math.abs(directionY)) {
            if (directionX > 0) {
                fingerDirection = ActionEvent.FINGER_DIRECTION_RIGHT;
            } else {
                fingerDirection = ActionEvent.FINGER_DIRECTION_LEFT;
            }
        } else {
            if (directionY > 0) {
                fingerDirection = ActionEvent.FINGER_DIRECTION_DOWN;
            } else {
                fingerDirection = ActionEvent.FINGER_DIRECTION_UP;
            }
        }

        return fingerDirection;
    }

    private void initiateVelocityTracker(MotionEvent event) {
        if (mVelocityTracker == null) {
            // Retrieve a new VelocityTracker object to watch the velocity of a motion.
            mVelocityTracker = getVelocityTracker();
        } else {
            mVelocityTracker.clear();
        }
        // Add a user's movement to the tracker.
        mVelocityTracker.addMovement(event);
    }

    private void recycleVelocityTracker() {
        // This may have been cleared already.
        if (mVelocityTracker != null) {
            mVelocityTracker.recycle();
            mVelocityTracker = null;
        }
    }
}
