package com.contentsquare.android.internal.ui.processor;

import android.support.annotation.NonNull;
import android.view.View;
import android.view.ViewGroup;

/**
 * A Visitor pattern implementation which can traverse a {@link android.view.ViewGroup} and
 * execute actions on every {@link View} or {@link ViewGroup} it lands on.
 */
public final class LayoutTraverser {

    @NonNull
    private final Processor mProcessor;

    private LayoutTraverser(@NonNull Processor processor) {
        this.mProcessor = processor;
    }

    /**
     * Creates a layout traverse instance with a provided a {@link Processor}.
     *
     * @param processor the processor for the views traversed.
     */
    @NonNull
    public static LayoutTraverser build(@NonNull Processor processor) {
        return new LayoutTraverser(processor);
    }

    /**
     * Traverses a {@link ViewGroup} by visiting all children and executing actions defined in the
     * {@link Processor} provided by the {@link #build(Processor)} pattern.
     *
     * @param root the root view of the hierarchy to be traversed.
     */
    public void traverse(@NonNull ViewGroup root) {
        mProcessor.process(root);
        final int childCount = root.getChildCount();
        for (int i = 0; i < childCount; ++i) {
            final View child = root.getChildAt(i);
            if (child instanceof ViewGroup) {
                traverse((ViewGroup) child);
            } else {
                mProcessor.process(child);
            }
        }
    }

    /**
     * An interface which define a processor which will react on one of the two types which can be
     * crossed traversing the hierarchy.
     * This interface should be defined by the caller.
     */
    public interface Processor {
        /**
         * Process a {@link View}. This method will be called when the {@link LayoutTraverser}
         * reaches a view instance primary targets are widgets.
         *
         * @param view the view in process.
         */
        void process(@NonNull View view);

        /**
         * Process a {@link ViewGroup}. This method will be called when the {@link LayoutTraverser}
         * reaches a viewGroup instance primary targets are Layouts.
         *
         * @param viewGroup the viewgroup in process
         */
        void process(@NonNull ViewGroup viewGroup);
    }

}
