package com.contentsquare.android.internal.ui.processor.descriptors;

import static com.contentsquare.android.internal.util.ResourceUtils.EMPTY_STRING_ID;
import static com.contentsquare.android.internal.util.ResourceUtils.getResourceEntryName;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;
import android.view.ViewGroup;

import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.screengraph.TreeTraverser;
import com.contentsquare.android.internal.util.Strings;

import java.util.Locale;

/**
 * This class generates paths from the root of the
 * {@link android.support.v4.view.ViewPager.DecorView}
 * to the leaf we're providing in various formats.
 */
public class PathDescriptor {

    /**
     * Generates the Analytics String path for a view starting from the root.
     *
     * @param view as the {@link View} to be processed
     * @return {@link String} as - the full path ot the view
     */
    @NonNull
    public String generateAnalyticsPath(@Nullable View view) {
        StringBuilder data = new StringBuilder("[root]>");
        if (view != null) {
            processLeaf(view, data);
        }
        // remove the trailing '>' at the end as it's too "difficult" to do it in the recursive call
        data.deleteCharAt(data.length() - 1);
        return data.toString();
    }

    private void processLeaf(@NonNull View view, @NonNull StringBuilder data) {
        ViewGroup parent = null;
        if (view.getParent() instanceof ViewGroup) {
            parent = (ViewGroup) view.getParent();
        } else {
            new Logger("path").w("discarding parent as it's not a ViewGroup");
        }
        if (parent == null || parent.getClass().toString().endsWith(
                "widget.FitWindowsLinearLayout")) {
            // return prematurely if the parent is null or the DecorView it's self.
            return;
        }
        String className = view.getClass().getSimpleName();
        processLeaf(parent, data);
        if (!className.startsWith(TreeTraverser.CS_LAYOUT_NAME)) {
            // have to skip this in the path segment, but have to continue processing it.
            data.append(className);
        }
        for (int itemPos = 0; itemPos < parent.getChildCount(); itemPos++) {
            View child = parent.getChildAt(itemPos);
            if (child == view) {
                String id = getResourceEntryName(child, EMPTY_STRING_ID);
                if (!id.contains(TreeTraverser.CS_GLASS)) {
                    // we completely skip this view's adding to the path, because we don't want it.
                    data.append(String.format(Locale.getDefault(), ":eq(%d)", itemPos));
                    if (!Strings.isNullOrEmpty(id)) {
                        // do not append a # if there's no ID.
                        data.append('#').append(id);
                    }
                    data.append('>');
                }
                break;
            }
        }

    }
}
