package com.contentsquare.android.internal.ui.processor.descriptors;

import static com.contentsquare.android.internal.dagger.SingletonProvider.getAppComponent;
import static com.contentsquare.android.internal.util.ResourceUtils.getResourceEntryName;

import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import com.contentsquare.android.internal.DeviceInfo;
import com.contentsquare.android.internal.logging.Logger;

/**
 * Descriptor for a {@link android.view.View}
 */
public abstract class PlaneDescriptor {
    @VisibleForTesting
    final PathDescriptor mPathDescriptor = new PathDescriptor();
    private final Logger mLogger = new Logger("PlaneDescriptor");
    private int mTopX;
    private int mTopY;
    private int mBottomX;
    private int mBottomY;
    private boolean mEnabled;
    private String mClassName;
    private CharSequence mAccessibilityClassName;
    private boolean mActivated;
    private boolean mClickable;
    private boolean mFocusable;
    private boolean mFocused;
    private boolean mIsShown;
    private String mViewResourceId;
    private int mViewId;
    private int mChildCount = -1;
    private boolean mTouchBlockingFocus = false;
    private boolean mVisible;
    private String mViewLabel;
    private String mViewAccessibilityLabel;
    private String mTargetPathDescriptor;

    PlaneDescriptor(@NonNull View view) {
        int[] top = new int[2];
        view.getLocationOnScreen(top);
        mTopX = top[0];
        mTopY = top[1];
        DeviceInfo deviceInfo = getAppComponent().getDeviceInfo();
        final int statusBarHeight = deviceInfo.getStatusBarHeight();
        final int actionBarHeight = deviceInfo.calculateActionBarHeight();


        mBottomX = mTopX + view.getWidth();
        mBottomY = mTopY + view.getHeight();
        mTopY -= statusBarHeight;
        mTopY -= actionBarHeight;
        mBottomY -= statusBarHeight;
        mBottomY -= actionBarHeight;

        String classname;
        try {
            classname = view.getParent().getClass().getSimpleName();

            mClassName = classname;
            mViewResourceId = getResourceEntryName((View) view.getParent(), null);
        } catch (NullPointerException e) {
            mLogger.e(e, "Failed to get class name");
        }
        if (view instanceof TextView) {
            CharSequence text = ((TextView) view).getText();
            mViewLabel = String.valueOf(text);
        }
        CharSequence description = view.getContentDescription();
        mViewAccessibilityLabel = String.valueOf(description);
        mTargetPathDescriptor = mPathDescriptor.generateAnalyticsPath(view);
    }

    /**
     * Returns an instance of PlaneDescriptor of the corresponding type depending of the instance of
     * the view, being a group or not.
     *
     * @param targetView the target view we need a descriptor for.
     * @return a corresponding descriptor.
     */
    @NonNull
    public static PlaneDescriptor getDescriptorFor(@NonNull View targetView) {
        if (targetView instanceof ViewGroup) {
            return new ViewGroupDescriptor((ViewGroup) targetView);
        } else {
            return new ViewDescriptor(targetView);
        }
    }


    public int getTopX() {
        return mTopX;
    }

    public int getTopY() {
        return mTopY;
    }

    public int getBottomX() {
        return mBottomX;
    }

    public int getBottomY() {
        return mBottomY;
    }

    public boolean isEnabled() {
        return mEnabled;
    }

    void setEnabled(boolean enabled) {
        mEnabled = enabled;
    }

    public String getClassName() {
        return mClassName;
    }

    void setClassName(String className) {
        mClassName = className;
    }

    public CharSequence getAccessibilityClassName() {
        return mAccessibilityClassName;
    }

    void setAccessibilityClassName(CharSequence accessibilityClassName) {
        mAccessibilityClassName = accessibilityClassName;
    }

    public boolean isActivated() {
        return mActivated;
    }

    void setActivated(boolean activated) {
        mActivated = activated;
    }

    public boolean isClickable() {
        return mClickable;
    }

    void setClickable(boolean clickable) {
        mClickable = clickable;
    }

    public boolean isFocusable() {
        return mFocusable;
    }

    void setFocusable(boolean focusable) {
        mFocusable = focusable;
    }

    public boolean isFocused() {
        return mFocused;
    }

    void setFocused(boolean focused) {
        mFocused = focused;
    }

    public boolean isShown() {
        return mIsShown;
    }

    void setShown(boolean shown) {
        mIsShown = shown;
    }

    public int getViewId() {
        return mViewId;
    }

    void setViewId(int viewId) {
        mViewId = viewId;
    }

    public String getViewResourceId() {
        return mViewResourceId;
    }

    void setViewResourceId(String viewResourceId) {
        mViewResourceId = viewResourceId;
    }

    public int getChildCount() {
        return mChildCount;
    }

    void setChildCount(int childCount) {
        mChildCount = childCount;
    }

    boolean isTouchBlockingFocus() {
        return mTouchBlockingFocus;
    }

    void setTouchBlockingFocus(boolean touchBlockingFocus) {
        mTouchBlockingFocus = touchBlockingFocus;
    }

    public boolean isVisible() {
        return mVisible;
    }

    public void setVisible(boolean visible) {
        mVisible = visible;
    }

    @NonNull
    public String getTargetPathDescriptor() {
        return mTargetPathDescriptor;
    }

    public String getViewLabel() {
        return mViewLabel;
    }

    public String getViewAccessibilityLabel() {
        return mViewAccessibilityLabel;
    }
}
