package com.contentsquare.android.internal.util;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.logging.Logger;

import java.io.IOException;
import java.io.InputStream;

/**
 * Utility class for interaction with the assets Folder.
 */
public class AssetManager {
    private static final String BASE_CONFIG = "baseconfig.json";
    private static final Logger LOG = new Logger("AssetManager");
    private final Context mContext;


    /**
     * Constructs an instance of the asset manager.
     * *MANAGED BY DAGGER*
     *
     * @param context the application context which is to be used.
     */
    public AssetManager(@NonNull final Context context) {
        mContext = context;
    }


    /**
     * Reads the content of the designated file from the assets folder, in a {@link String}.
     *
     * @param filename the name of the file
     * @return the content of the file or null if an exception occurred
     */
    @VisibleForTesting
    @Nullable
    String loadFileFromAsset(@NonNull String filename) {
        String fileContent;
        try {
            InputStream is = mContext.getAssets().open(filename);
            int size = is.available();
            byte[] buffer = new byte[size];
            int readBuffer = is.read(buffer);
            LOG.d("Bytes read: %d", readBuffer);
            is.close();
            fileContent = new String(buffer, "UTF-8");
        } catch (IOException ex) {
            LOG.e(ex, "Failed to read file from assets.");
            return null;
        }
        return fileContent;
    }

    /**
     * Reads the content of baseconfig.json
     *
     * @return {@link String} - the content of the file or null if an exception occurred
     */
    @Nullable
    public String loadBaseConfigFromAsset() {
        return loadFileFromAsset(BASE_CONFIG);
    }

}
