package com.contentsquare.android.internal.util;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.BuildConfig;
import com.contentsquare.android.internal.ConfigurationCompositor;
import com.contentsquare.android.internal.dagger.SingletonProvider;
import com.contentsquare.android.internal.logging.Logger;
import com.contentsquare.android.internal.preferences.PrefsHelper;

import org.json.JSONException;
import org.json.JSONObject;

import javax.inject.Inject;

/**
 * Helper class around handling the retrieval of configurations.
 */
public class ConfigurationHelper {
    /**
     * String used in shared preferences for storing the last retrieved configuration.
     */
    public static final String LAST_CONFIG_LABEL = "config";
    /**
     * String used as a label in shared preferences for storing the last retrieved configuration
     * with a ttl.
     */
    public static final String LAST_KNOWN_CONFIG = "last_config";
    /**
     * String used as a label in shared preferences for storing the last retrieved uid.
     */
    public static final String USERID = "uid";
    /**
     * String used in shared preferences for storing the last uid configuration with a ttl.
     */
    public static final String USERID_CONFIG = "uid_config";
    static final String TIMESTAMP = "timestamp";
    static final long TTL = BuildConfig.DEFAULT_CONFIG_TTL;
    static final long USERID_TTL = BuildConfig.USERID_CONFIG_TTL;
    private final long mNow = System.currentTimeMillis();

    @Inject
    @VisibleForTesting
    PrefsHelper mPrefsHelper;
    @Inject
    @VisibleForTesting
    ConfigurationCompositor mConfigurationCompositor;
    private final Logger mLogger = new Logger("ConfigurationHelper");

    /**
     * Constructor which initializes the mocks.
     */
    public ConfigurationHelper() {
        SingletonProvider.getAppComponent().inject(this);
    }

    /**
     * Provides the last valid config from the client. Validity is defined by TTL.
     */
    @Nullable
    public String getLastClientConfig() {
        String serializedJson = mPrefsHelper.getString(LAST_KNOWN_CONFIG, null);
        mLogger.d("retrieving last config from preferences");
        if (Strings.isNullOrEmpty(serializedJson)) {
            mLogger.d("last config is null");
            return null;
        }
        try {
            JSONObject configWithTtl = new JSONObject(serializedJson);
            long timestamp = configWithTtl.getLong(TIMESTAMP);
            if (mNow - timestamp > TTL) {
                // config is no longer valid, returning Null.
                mLogger.d("last config is outdated, returning null");
                mPrefsHelper.deleteAndResetAll();
                return null;
            } else {
                //valid config, return it
                mLogger.d("last config is valid, returning config from preferences");
                final String jsonConfig = configWithTtl.getString(LAST_CONFIG_LABEL);
                if (mConfigurationCompositor.validateConfig(jsonConfig)) {
                    return jsonConfig;
                } else {
                    return null;
                }
            }
        } catch (JSONException e) {
            mLogger.e(e, "failed to deserialize last config with an exception");
            return null;
        }
    }

    /**
     * Stores last config for this client, with a timestamp for the save.
     *
     * @param config the server configuration.
     */
    public void setLastConfig(@Nullable String config) {
        if (Strings.isNullOrEmpty(config)) {
            return;
        }
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put(LAST_CONFIG_LABEL, config);
            jsonObject.put(TIMESTAMP, System.currentTimeMillis());
            mPrefsHelper.putString(LAST_KNOWN_CONFIG, jsonObject.toString());
        } catch (JSONException e) {
            mLogger.e(e, "Failed to serialize and store the config.");
        }
    }

    /**
     * Stores last uid config with a timestamp.
     *
     * @param uid the server configuration.
     */
    public void setUserIdConfig(@NonNull String uid) {

        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put(USERID, uid);
            jsonObject.put(TIMESTAMP, System.currentTimeMillis());
            mPrefsHelper.putString(USERID_CONFIG, jsonObject.toString());
            mLogger.d("Saving USER ID config to sharedPrefs.");
        } catch (JSONException e) {
            mLogger.e(e, "Failed to serialize and store the USER ID config.");
        }
    }

    /**
     * Provides the latest valid uid config from the client. Validity is defined by USERID_TTL.
     */
    @Nullable
    public String getLastUserIdConfig() {
        String serializedJson = mPrefsHelper.getString(USERID_CONFIG, null);
        mLogger.w("retrieving last USER ID config from preferences");
        if (Strings.isNullOrEmpty(serializedJson)) {
            mLogger.w("last USER ID config is null");
            return null;
        }
        try {
            JSONObject configWithTtl = new JSONObject(serializedJson);
            long timestamp = configWithTtl.getLong(TIMESTAMP);
            if (mNow - timestamp > USERID_TTL) {
                mLogger.w("last USER ID is outdated, returning null");
                mPrefsHelper.deleteAndResetAll();
                return null;
            } else {
                mLogger.w("last USER ID is valid, returning USER ID from preferences %s",
                        configWithTtl.getString(USERID));
                return configWithTtl.getString(USERID);
            }
        } catch (JSONException e) {
            mLogger.e(e, "failed to deserialize last USER ID config with an exception");
            return null;
        }
    }

}
