package com.contentsquare.android.internal.util;

import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.contentsquare.android.internal.logging.Logger;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.LinkedList;
import java.util.List;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

/**
 * Utility class that contains basic file storage functions.
 */
public class FileStorageUtil {

    private static final String DATA_CONTAINER = "ContentSquare";
    private final Logger mLogger = new Logger("FileStorageUtil");
    @VisibleForTesting
    String mAppFilesLocation = null;

    /**
     * FileStorageUtil constructor.
     */
    public FileStorageUtil() {
        //Default constructor
    }

    /**
     * Create the ContentSquare folder which contains session.
     *
     * @param sessionId the session id
     * @return boolean of whether a new file is created
     */
    public boolean prepareAndReset(@IntRange(from = 0) int sessionId, @NonNull String mainDirPath) {
        mAppFilesLocation = mainDirPath;
        return createStorageFolderIfNotExist(String.valueOf(sessionId));
    }

    /**
     * Checks the internal storage for an active file for session. Create file if it doesn't exists.
     *
     * @param folder the session id
     * @param file   the file id
     */
    public boolean prepareFile(@NonNull String folder, @NonNull String file) {
        return createFileIfNotExist(folder, file);
    }

    /**
     * Appends a new line of event to the current active file.
     *
     * @param folderName the session id
     * @param filename   the file id
     * @param data       the event data to be appended
     */
    public void appendStringToFile(@NonNull String folderName, @NonNull String filename,
            @NonNull String data) {
        File file = getFile(getFolderPath(folderName)
                .append(File.separator)
                .append(filename).toString());
        String dataString = data.replace("\n", "") + "\n";

        if (file.exists()) {
            try {
                FileOutputStream fileOutputStream = getFileOutputStream(file, true);
                fileOutputStream.write(dataString.getBytes(Charset.forName(Strings.UTF_8)));
                fileOutputStream.close();
                mLogger.d("Writting to File: %s", dataString);

            } catch (IOException e) {
                mLogger.e(e, " Data not written to file");
            }
        }
    }

    /**
     * Appends an event to the current active file.
     *
     * @param folderName the session id
     * @param filename   the file id
     * @return list of {@link String} events read from file
     */
    @NonNull
    public List<String> readFileContentByLine(@NonNull String folderName,
            @NonNull String filename) {

        File file = getFile(getFolderPath(folderName)
                .append(File.separator)
                .append(filename).toString());
        List<String> listEvents = new LinkedList<>();

        FileInputStream fileInputStream = null;
        BufferedReader reader;

        if (file.exists()) {
            try {
                fileInputStream = getFileInputStream(file);
                reader = getBufferReader(getInputStreamReader(fileInputStream));
                //CHECKSTYLE:OFF
                for (String line; (line = reader.readLine()) != null; ) {
                    //CHECKSTYLE:ON
                    mLogger.d("File Data: %s", line);
                    listEvents.add(line);
                }
            } catch (NullPointerException | IOException e) {
                mLogger.e(e, "Failed while reading file : %s/%s", folderName, filename);
            } finally {
                try {
                    if (fileInputStream != null) {
                        fileInputStream.close();
                    }
                } catch (IOException e) {
                    mLogger.w(e, "Failed to close stream");
                }
            }

        }

        return listEvents;
    }

    /**
     * Gets the amount of files in a folder.
     *
     * @param folderName the session id
     * @return the number of File in the current folder
     */
    public int getNumberOfItemsInFolder(@NonNull String folderName) {
        File folder = getFile(getFolderPath(folderName).toString());
        String[] lister = folder.list();
        // lister could be null if the system encounters an IO error or the folder doesn't exist.
        if (lister == null) {
            return 0;
        } else {
            return lister.length;
        }
    }

    /**
     * Deletes a specific file.
     *
     * @param folderName the session id
     * @param filename   the file id
     * @return success of this op.
     */
    public boolean deleteFile(@NonNull String folderName, @NonNull String filename) {
        File file = getFile(getFolderPath(folderName)
                .append(File.separator)
                .append(filename).toString());

        mLogger.w("File to delete: %s", file.getPath());
        return file.delete();
    }

    private boolean createStorageFolderIfNotExist(@NonNull String folderName) {
        File folder = getFile(getFolderPath(folderName).toString());
        return !folder.exists() && folder.mkdirs();
    }

    private StringBuilder getFolderPath(String folderName) {
        return new StringBuilder()
                .append(mAppFilesLocation)
                .append(File.separator)
                .append(DATA_CONTAINER)
                .append(File.separator)
                .append(folderName);
    }

    private boolean createFileIfNotExist(@NonNull String folderName, @Nullable String filename) {
        File file = getFile(getFolderPath(folderName)
                .append(File.separator)
                .append(filename).toString()
        );

        if (!file.exists()) {
            try {
                return file.createNewFile();
            } catch (IOException e) {
                mLogger.e(e, "Error creating File %s ", file.getPath());
                return false;
            }
        } else {
            mLogger.d("File exists %s ", file.getPath());
            return true;
        }
    }

    @VisibleForTesting
    File getFile(String path) {
        return new File(path);
    }

    @VisibleForTesting
    FileOutputStream getFileOutputStream(File file, boolean append)
            throws FileNotFoundException {
        return new FileOutputStream(file, append);
    }

    @VisibleForTesting
    FileInputStream getFileInputStream(File file) throws FileNotFoundException {
        return new FileInputStream(file);
    }

    @VisibleForTesting
    InputStreamReader getInputStreamReader(FileInputStream fileInputStream) {
        return new InputStreamReader(fileInputStream, Charset.forName(Strings.UTF_8));
    }

    @VisibleForTesting
    BufferedReader getBufferReader(InputStreamReader inputStreamReader) {
        return new BufferedReader(inputStreamReader);
    }

    /**
     * Gets the location where all ContentSquare data is saved.
     *
     * @return the path of the location of all ContentSquare data.
     */
    @NonNull
    public String[] getMainDirectoryList() {
        String[] fileNames;
        String mainDir = mAppFilesLocation + File.separator + DATA_CONTAINER;

        File directory = getFile(mainDir);

        mLogger.d("Directory: %s", directory.getName());
        if (directory.list() == null) {
            mLogger.e("Can't access Folder : %s", directory.getName());
            return new String[0];
        } else {
            File[] files = directory.listFiles();
            if (files != null && files.length > 0) {
                fileNames = new String[files.length];
                for (int i = 0; i < files.length; i++) {
                    fileNames[i] = files[i].getAbsolutePath();
                }
                return fileNames;
            } else {
                return new String[0];
            }
        }

    }

    /**
     * Gets list of {@link File} from a directory.
     *
     * @param directory the folder where should be look into.
     * @return array of {@link File} containing all the files from the directory.
     */
    @NonNull
    public File[] getFileList(@NonNull File directory) {
        mLogger.d("Directory: %s", directory.getName());
        return directory.listFiles();
    }

    /**
     * Deletes an empty folder.
     *
     * @param folderName the path of the folder to be cleaned up
     */
    @SuppressFBWarnings(value = "RV_RETURN_VALUE_IGNORED_BAD_PRACTICE")
    public void cleanupFolder(@NonNull String folderName) {

        String dir =
                mAppFilesLocation + File.separator + DATA_CONTAINER + File.separator + folderName;

        File folder = getFile(dir);
        String[] content = folder.list();

        mLogger.w("Folder to delete: %s", folder.getPath());

        if (content != null && content.length == 0) {
            boolean deleted = folder.delete();
            mLogger.w("folder %s deleted : %b", folderName, deleted);
        }
    }

    /**
     * Verifies if it is a directory.
     *
     * @param directory file corresponding to the directory to be verified
     * @return whether its directory
     */
    public boolean isDirectory(@NonNull File directory) {

        boolean isDir = false;
        if (directory.isDirectory()) {
            isDir = true;
        }
        return isDir;
    }

    /**
     * Verifies if a File/Directory is readable.
     *
     * @param directory file corresponding to the directory to read
     * @return whether directory is readable
     */
    public boolean readable(@NonNull File directory) {

        boolean isReadable = false;
        if (directory.canRead()) {
            isReadable = true;
        }
        return isReadable;
    }

}
