package com.contentsquare.android.internal.util;

import static com.contentsquare.android.internal.util.Strings.UTF_8;
import static com.contentsquare.android.internal.util.Strings.encode;
import static java.net.HttpURLConnection.HTTP_OK;

import android.net.Uri;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StringDef;
import android.support.annotation.VisibleForTesting;
import android.util.Pair;

import com.contentsquare.android.internal.logging.Logger;

import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.List;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

/**
 * Implementation providing the means of making Http POST and GET requests from the ContentSquare
 * SDK. This implementation is making use of {@link java.net.HttpURLConnection}.
 */
public class HttpConnection {

    static final String GET = "GET";
    static final String POST = "POST";
    static final String HEAD = "HEAD";
    static final String OPTIONS = "OPTIONS";
    static final String PUT = "PUT";
    static final String DELETE = "DELETE";
    static final String TRACE = "TRACE";
    static final int TIMEOUT_CONNECT = 1_000; // 1s
    private static final int TIMEOUT_READ = 10_000; // 15s
    private static final char EQUALS = '=';
    private static final char AND = '&';
    private final Logger mLog = new Logger("HttpConnection");

    /**
     * Creates and executes an Http Get Request with a set of args provided in the form of a
     * list of {@link Pair}s.
     *
     * @param endpoint A String representation of the URL you want to call.
     * @param args     a list of the key-values you want to send.
     * @return a {@link HttpResponse} representing the server response.
     */
    @SuppressFBWarnings(value = "SIC_INNER_SHOULD_BE_STATIC",
            justification = "We are OK with this not being a static class.")
    @NonNull
    public HttpResponse performGetWithFormdata(@NonNull String endpoint,
                                               @Nullable List<Pair> args) {
        HttpURLConnection connection = null;
        HttpResponse response = new HttpResponse();
        try {
            String requestUrl = buildGetUrl(endpoint, args);
            connection = openConnection(GET, requestUrl, null);

            //Send request
            connection.connect();

            //Get Response
            response.setStatus(connection.getResponseCode());
            mLog.d("GET: Connection open, status %d, url : %s ", response.getStatus(), requestUrl);

            if (response.getStatus() == HTTP_OK) {
                response.setStringResponse(readStream(connection.getInputStream()));
            }
        } catch (IOException | NullPointerException e) {
            mLog.e(e, "Exception while processing HttpGet Request on %s ", endpoint);
            response.setStatus(HttpURLConnection.HTTP_INTERNAL_ERROR);
            response.setException(e);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
        return response;
    }

    /**
     * Creates and executes an Http Post Request with a {@link JSONObject}.
     *
     * @param endpoint A String representation of the URL you want to call.
     * @param json     a list of the key-values you want to send.
     * @return a {@link HttpResponse} representing the server response.
     */
    @NonNull
    public HttpResponse performPostWithJson(@NonNull String endpoint, @NonNull JSONObject json) {
        mLog.i("Performing post on %s with %s ", endpoint, json);
        HttpURLConnection connection = null;
        try {
            String requestData = json.toString();
            connection = openConnection(POST, endpoint, requestData);
            return performPost(connection, endpoint, requestData);
        } catch (IOException | NullPointerException e) {
            mLog.e(e, "Exception while processing HttpPOST Request on %s \n  for json: %s",
                    endpoint, json);
            return handleError(e);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    private HttpResponse handleError(Exception exception) {
        HttpResponse response = new HttpResponse();
        response.setStatus(HttpURLConnection.HTTP_INTERNAL_ERROR);
        response.setException(exception);
        return response;

    }

    /**
     * Executes a Generic Post with the provided arguments.
     *
     * @param connection    the connection to use for this record
     * @param endpoint      the url we're "posting" to
     * @param processedArgs the arguments for this post in a normalized format
     * @return a {@link HttpResponse} object representing this call's result
     * @throws IOException in the case the connection breaks or similar
     */
    @NonNull
    @VisibleForTesting
    HttpResponse performPost(
            @NonNull HttpURLConnection connection,
            @NonNull String endpoint,
            @NonNull String processedArgs)
            throws IOException {
        final HttpResponse response = new HttpResponse();

        //Send request
        OutputStream wr = connection.getOutputStream();
        wr.write(processedArgs.trim().getBytes(UTF_8));
        wr.flush();
        wr.close();

        //Get Response
        response.setStatus(connection.getResponseCode());
        mLog.d("POST: Connection open, status %s, url : %s",
                String.valueOf(response.getStatus()),
                endpoint);

        if (response.getStatus() == HTTP_OK) {
            response.setStringResponse(readStream(connection.getInputStream()));
        }
        mLog.i("Received response : %s", response.getStringResponse());
        return response;
    }

    /**
     * Opens a connection to a specific URL which is prepared for a post/get request following it.
     *
     * @param method      the type of request, should be one defined by a valid {@link HttpMethod}
     * @param endpoint    the server endpoint
     * @param encodedArgs the arguments for this request, http encoded
     * @return a prepared and opened connection
     * @throws IOException if any exception occurs during parsing the url or opening the
     *                     connection.
     */
    @NonNull
    @VisibleForTesting
    @SuppressWarnings("squid:S00112")
    HttpURLConnection openConnection(@NonNull @HttpMethod String method,
                                     @NonNull String endpoint,
                                     @Nullable String encodedArgs)
            throws IOException {
        //If we're thinking of making a check for internet connectivity, it should be done here.
        mLog.d("opening a %s connection to %s", method, endpoint);
        URL url = new URL(endpoint);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod(method);
        connection.setRequestProperty("Charset", UTF_8);
        connection.setUseCaches(false);
        connection.setAllowUserInteraction(false);
        connection.setConnectTimeout(TIMEOUT_CONNECT);
        connection.setReadTimeout(TIMEOUT_READ);

        switch (method) {
            case GET:
                connection.setRequestProperty("Content-length", "0");
                break;
            case POST:
                connection.setDoInput(true);
                connection.setDoOutput(true);
                connection.setRequestProperty("Content-Type", "application/json; "
                        + "charset="
                        + UTF_8);
                break;
            default:
                throw new RuntimeException(
                        "HTTP Methods other than POST and GET are not implemented.");

        }
        return connection;
    }

    /**
     * Builds a Http GET url with parameters.
     */
    @VisibleForTesting
    String buildGetUrl(@NonNull String url, @Nullable List<Pair> arguments) {
        if (arguments == null) {
            return url;
        }
        Uri.Builder builder = Uri.parse(url).buildUpon();
        for (Pair pair : arguments) {
            builder.appendQueryParameter(
                    pair.first.toString(),
                    pair.second.toString()
            );
        }
        String derivedUrl = builder.build().toString();
        mLog.d("Get URL transformed to %s", url, "");
        return derivedUrl;
    }

    /**
     * Process arguments and build a string representation of a post request format.
     */
    @Nullable
    @VisibleForTesting
    String urlEncodeRequestParameters(@Nullable List<Pair> arguments) {
        if (arguments == null) {
            return null;
        }
        StringBuilder builder = new StringBuilder();
        int size = arguments.size();

        for (int i = 0; i < size; i++) {
            Pair pair = arguments.get(i);
            builder
                    .append(encode(String.valueOf(pair.first)))
                    .append(EQUALS)
                    .append(encode(String.valueOf(pair.second)));
            if (i < size - 1) {
                builder.append(AND);
            }
        }
        return builder.toString();
    }

    /**
     * Reads an input stream in a String Builder and prints it in a string.
     */
    @VisibleForTesting
    String readStream(InputStream inputStream) throws IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(inputStream, UTF_8));
        StringBuilder sb = new StringBuilder();
        String line;
        while ((line = br.readLine()) != null) {
            sb.append(line).append('\n');
        }
        br.close();
        return sb.toString();
    }

    @Retention(RetentionPolicy.SOURCE)
    @StringDef( {GET, POST, HEAD, OPTIONS, PUT, DELETE, TRACE})
    @interface HttpMethod {
    }

    /**
     * This class represents the response of an Http Call.
     */
    public class HttpResponse {
        private int mStatus = -1;
        private String mStringResponse = null;
        private Throwable mException = null;

        /**
         * Constructor which is primarily used in the case of a successful http call.
         */
        HttpResponse() {
        }

        /**
         * Gets the status of an executed Http call, or -1 if there was an error.
         *
         * @return the status
         */
        public int getStatus() {
            return mStatus;
        }

        void setStatus(int status) {
            mStatus = status;
        }

        /**
         * Gets the String representation of the server response.
         *
         * @return the response
         */
        @Nullable
        public String getStringResponse() {
            return mStringResponse;
        }

        void setStringResponse(String stringResponse) {
            mStringResponse = stringResponse;
        }

        /**
         * Gets the exception which happened during transmission, or null.
         *
         * @return the exception or null
         */
        @Nullable
        public Throwable getException() {
            return mException;
        }

        public void setException(Throwable exception) {
            mException = exception;
        }

        /**
         * Validates this request has been successful in sending data.
         *
         * @return true if the status was {@link HttpURLConnection#HTTP_OK}
         */
        public boolean positive() {
            return getStatus() == HTTP_OK;
        }
    }
}
