package com.contentsquare.android.internal.util;

import android.os.Build;

/**
 * Math Utility containing methods to calculate point and plane geometry.
 */

public class MathUtil {
    /**
     * Avoid instance creation.
     */
    private MathUtil() {
    }

    /**
     * Calculates the position of a point defined by X and Y in a plan defined by ancor, width and
     * height.
     *
     * @param planeLeft   the x position of the plane anchor.
     * @param planeTop    the y position of the plane anchor.
     * @param planeWidth  the width of the plane calculated from the anchor.
     * @param planeHeight the height of the plane calculated from the anchor.
     * @param pointX      the X coordinate of the point of interest.
     * @param pointY      the X coordinate of the point of interest.
     * @return true if the point is in the plane, false otherwise.
     */
    public static boolean isPointInPlane(
            int planeLeft, int planeTop, int planeWidth, int planeHeight,
            int pointX, int pointY) {
        // calculations are done from the top left corner being 0,0 towards the bottom right being
        // considered the end with width, height
        return (pointX >= planeLeft) // to right of left most
                && (pointX <= (planeLeft + planeWidth)) // to left of right most
                && (pointY >= planeTop) // below start
                && (pointY <= (planeTop + planeHeight));//above bottom
    }

    /**
     * Calculates the Euclidean distance which is an "ordinary" straight-line distance between two
     * points in space.
     *
     * @param startX starting point X coordinates
     * @param startY Starting point Y coordinates
     * @param endX   ending point X coordinates
     * @param endY   ending point Y coordinates
     * @return the distance between start and end.
     */
    public static double euclideanDistance(
            final float startX,
            final float startY,
            final float endX,
            final float endY) {

        final float xDistance = endX - startX;
        final float yDistance = endY - startY;
        return Math.sqrt(xDistance * xDistance + yDistance * yDistance);
    }

    /**
     * Calculates the area of a plane.
     *
     * @param width  of the plane
     * @param height of the plane
     * @return the area of the plane
     */
    static int calculateArea(final int width, final int height) {
        return width * height;
    }

    /**
     * This is a wrapper method around {@link Integer#compare(int, int)}. it's a direct call on API
     * 19, or a copy of it for SDKs below API19.
     *
     * @param first  the first int
     * @param second the second int
     * @return the value {@code 0} if {@code x == y};
     * a value less than {@code 0} if {@code x < y}; and
     * a value greater than {@code 0} if {@code x > y}
     */
    public static int compareInts(int first, int second) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            return Integer.compare(second, first);
        } else {
            // this is the implementation which is default on android,
            // we have to have this block as it  was implemented in API 19 and
            // we offer support starting with api 15.
            //CHECKSTYLE:OFF
            int x = second;
            int y = first;
            return (x < y) ? -1 : ((x == y) ? 0 : 1);
            //CHECKSTYLE:ON
        }

    }

    /**
     * Safely converts a long to an int.
     * If your provided parameter is greater or smaller than {@link Integer#MAX_VALUE},
     * respectively {@link Integer#MIN_VALUE} the returned value will be truncated to one of
     * those limits.
     *
     * @param valueToConvert as long
     * @return the converted value as int
     */
    public static int safeLongToInt(long valueToConvert) {
        return (int) Math.max(Math.min(valueToConvert, Integer.MAX_VALUE), Integer.MIN_VALUE);
    }

    /**
     * Safely converts a float to an int.
     * If your provided parameter is greater or smaller than {@link Integer#MAX_VALUE},
     * respectively {@link Integer#MIN_VALUE} the returned value will be truncated to one of
     * those limits.
     *
     * @param valueToConvert as float
     * @return the converted value as int
     */
    public static int safeFloatToInt(float valueToConvert) {
        final float half = 0.5f;
        return (int) (valueToConvert + half);
    }
}
