package com.contentsquare.android.internal.util;

import android.content.Context;
import android.content.res.Resources;
import android.support.annotation.IdRes;
import android.support.annotation.NonNull;
import android.view.View;

/**
 * Utilities related to working with {@link android.content.res.Resources}.
 */

public class ResourceUtils {

    /**
     * Empty string descriptor with a null value.
     */
    public static final String NULL_STRING_ID = "null";
    /**
     * Empty string descriptor with no value (empty).
     */
    public static final String EMPTY_STRING_ID = "";

    @IdRes
    private static final long NEGATIVE_FORMAT = 0xFFFFFF;
    @IdRes
    private static final long ZERO_FORMAT = 0x000000;

    /**
     * Preserving encapsulation.
     */
    private ResourceUtils() {
    }

    /**
     * Returns resource entry name regarding to the {@link View}.
     *
     * @param view the view currently in evaluation.
     * @return resource entry name if the {@link View} is not null nor 0xffffffff.
     */
    @NonNull
    public static String getResourceEntryName(@NonNull View view, String defaultVal) {
        return getResourceEntryName(view.getContext(), view.getId(), defaultVal);
    }

    /**
     * Returns resource entry name regarding to the {@link View}.
     *
     * @param viewId     the view currently in evaluation.
     * @param defaultVal the default value to return if there is an error
     * @return resource entry name if the {@link View} is not null nor 0xffffffff.
     */
    @SuppressWarnings("squid:S1166")
    @NonNull
    public static String getResourceEntryName(
            @NonNull Context context,
            @IdRes int viewId,
            String defaultVal) {
        String descriptor = defaultVal;

        if (viewId != NEGATIVE_FORMAT && viewId != ZERO_FORMAT && viewId != -1) {
            try {
                // CHECKSTYLE:OFF
                descriptor = context.getResources().getResourceEntryName(viewId);
                // CHECKSTYLE:ON
            } catch (Resources.NotFoundException | /*Samsung throws null*/ NullPointerException e) {
                // The view does not have an id assigned. We cannot do anything about that
                // unfortunately.
            }
        }
        return descriptor;
    }
}
